# -*- coding: utf-8 -*-
import logging
import os
import sandbox.projects.common.build.parameters as build_parameters
from sandbox import sdk2
from sandbox.sandboxsdk import errors
from sandbox.projects.resource_types import ARCADIA_BINARY_ARCHIVE

DEFAULT_EXPORT_CMD_ARGS = '--logging_level INFO'


class HaasEngTopologyExportToYt(sdk2.Task):
    """ Run me to upload haas/eng_topology/struct dir to YT. """

    _arcadia_path = 'arcadia_src'
    _patch_path = 'patch_src'
    _build_target = 'haas/eng_topology/app/export_to_yt'

    _param_arcadia_patch_placeholder = '{arcadia_patch}'

    class Parameters(sdk2.Parameters):
        checkout_arcadia_from_url = build_parameters.ArcadiaUrl()
        arcadia_patch = sdk2.parameters.String(
            build_parameters.ArcadiaPatch.description,
            default=build_parameters.ArcadiaPatch.default,
            multiline=True
        )
        use_last_binary = sdk2.parameters.Bool('Use last binary archive', default=True)
        resource = sdk2.parameters.Resource("Resource to make available via {resource} in the command. "
                                            "Leave blank if use_last_binary is checked.")
        export_cmd_args = sdk2.parameters.String(
            'Arguments to run export_to yt command with',
            default=DEFAULT_EXPORT_CMD_ARGS
        )
        vault_env = sdk2.parameters.Dict("Vault items to put in the environment")

    @property
    def arcadia_path(self):
        return self.path(self._arcadia_path)

    @property
    def patch_path(self):
        return self.path()

    def pre_execute(self):
        checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        try:
            checkout_arcadia_from_url = sdk2.svn.Arcadia.freeze_url_revision(checkout_arcadia_from_url)
        except errors.SandboxSvnError as error:
            raise errors.SandboxTaskUnknownError(
                'Arcadia URL {0} does not exist. Error: {1}'.format(checkout_arcadia_from_url, error)
            )

        parsed_url = sdk2.svn.Arcadia.parse_url(checkout_arcadia_from_url)
        self.Context.ap_arcadia_revision = parsed_url.revision
        self.Context.ap_arcadia_trunk = parsed_url.trunk
        self.Context.ap_arcadia_branch = parsed_url.branch
        self.Context.ap_arcadia_tag = parsed_url.tag
        self.Context.checkout_arcadia_from_url = checkout_arcadia_from_url

    def on_execute(self):
        logging.info('Hello, Sandbox!')
        self.pre_execute()

        # Найти бинарь, который будет экспортировать топологию в YT
        executable_resource = self._get_executable_resource()

        # Клонировать директорию с топологией
        logging.info('checkout_arcadia_from_url={}'.format(self.Context.checkout_arcadia_from_url))
        logging.info('arcadia_path={}'.format(self.arcadia_path))
        sdk2.svn.Arcadia.checkout(url=self.Context.checkout_arcadia_from_url,
                                  path=self.arcadia_path,
                                  revision=self.Context.ap_arcadia_revision)
        logging.info('arcadia_patch = {}'.format(self.Parameters.arcadia_patch))
        if self.Parameters.arcadia_patch and self.Parameters.arcadia_patch != self._param_arcadia_patch_placeholder:
            sdk2.svn.Arcadia.apply_patch(self.arcadia_path, self.Parameters.arcadia_patch, self.patch_path)

        logging.info('listdir {}'.format(self.arcadia_path))
        for x in os.listdir(str(self.arcadia_path)):
            logging.info(x)

        # Запустить экспорт
        self._run_command_export_to_yt(executable_resource)

    def _get_executable_resource(self):
        if self.Parameters.use_last_binary:
            build_resource = ARCADIA_BINARY_ARCHIVE.find(attrs={'target': self._build_target}).first()
            logging.info('build_resource={}'.format(build_resource))
            logging.info('build_resource.id={}'.format(build_resource.id))
            return build_resource
        logging.info('self.Parameters.resource={}'.format(self.Parameters.resource))

        return self.Parameters.resource

    def _run_command_export_to_yt(self, executable_resource):
        resource_data = sdk2.ResourceData(executable_resource)
        resource_path = str(resource_data.path)
        cmd = '{resource} --struct_dir "{struct_dir}" {cmd_args}' \
            .format(resource=resource_path, struct_dir=self.arcadia_path, cmd_args=self.Parameters.export_cmd_args)
        logging.info('Running cmd: {}'.format(cmd))

        command_env = os.environ.copy()
        for vault_item, env_name in self.Parameters.vault_env.items():
            command_env[env_name] = sdk2.Vault.data(self.owner, vault_item)

        with sdk2.helpers.ProcessLog(self, logger="command") as pl:
            sdk2.helpers.subprocess.check_call(
                cmd,
                env=command_env,
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stderr,
                close_fds=True
            )
