# -*- coding: utf-8 -*-
import logging
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.BuildDockerImageFromGit import BuildDockerImageFromGit


class HaasTermItdcRunTestenv(sdk2.Task):
    """ Run me to execute shell command in Terminal ITDC test-env container. """
    class Parameters(BuildDockerImageFromGit.Parameters):
        command = sdk2.parameters.String("Command to execute", required=True)

    class Requirements(sdk2.Task.Requirements):
        client_tags = (ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL)
        dns = ctm.DnsType.DNS64
        disk_space = 1024 * 50
        ram = 1024 * 16
        # LXC_CONTAINER_WITH_DOCKER https://sandbox.yandex-team.ru/resource/773239891
        container_resource = 773239891

    def on_execute(self):
        # Build image
        self._run_stage(stage_name='build_image', stage_callable=self._build_image)

        # Run command in image
        self._docker_login()
        self._run_testenv_cmd()

    def _run_stage(self, stage_name, stage_callable):
        context_subtask_id_attr = '{}_task_id'.format(stage_name)
        with getattr(self.memoize_stage, stage_name):
            logging.info('Run task {}'.format(stage_name))
            sub_task_id = stage_callable()
            setattr(self.Context, context_subtask_id_attr, sub_task_id)
            logging.info('Wait output from task {}'.format(sub_task_id))
            raise sdk2.WaitTask(sub_task_id, ctt.Status.Group.FINISH + ctt.Status.Group.BREAK, wait_all=True,
                                timeout=7200)

        sub_task_id = getattr(self.Context, context_subtask_id_attr)
        sub_task = sdk2.Task.find(id=sub_task_id, children=True, status=ctt.Status.SUCCESS) \
            .order(-sdk2.Task.id).first()
        if not sub_task:
            raise TaskFailure('Child task ({}) in bad state: {}'.format(stage_name, sub_task_id))

    def _build_image(self):
        """ Build test-env image and push it to registry """
        params = ['container', 'repository', 'branch', 'commit', 'use_lfs', 'filter_branches', 'update_submodules',
                  'path_to_dockerfile', 'dockerfile_name', 'make_archive', 'ssh_vault_name', 'ssh_vault_owner',
                  'registry_tags', 'oauth_vault_name', 'oauth_vault_owner', 'registry_login', 'cache_from_image_tag',
                  'docker_build_args']
        sub_task_kwargs = {_: getattr(self.Parameters, _) for _ in params}
        sub_task = BuildDockerImageFromGit(
            self,
            description='Child of task {}: {}'.format(self.id, 'build_image'),
            # notifications=self.Parameters.notifications,
            create_sub_task=True,
            **sub_task_kwargs
        )
        sub_task.Requirements.ram = 1024 * 8
        sub_task.save().enqueue()

        return sub_task.id

    def _run_testenv_cmd(self):
        with sdk2.helpers.ProcessLog(self, logger="testenv_cmd") as pl:
            sdk2.helpers.subprocess.check_call(self.Parameters.command, shell=True, stdout=pl.stdout, stderr=pl.stderr,
                                               close_fds=True)

    def _docker_login(self):
        with sdk2.helpers.ProcessLog(self, logger="docker_login") as pl:
            sdk2.helpers.subprocess.check_call(self._docker_login_cmd(), shell=True, stdout=pl.stdout, stderr=pl.stderr,
                                               close_fds=True)

    def _docker_login_cmd(self):
        oauth_token = sdk2.Vault.data(self.Parameters.oauth_vault_owner, self.Parameters.oauth_vault_name)
        return 'docker login -u {login} -p {password} registry.yandex.net'.format(login=self.Parameters.registry_login,
                                                                                  password=oauth_token)
