# -*- coding: utf-8 -*-

import os
import shutil

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import error_handlers as eh

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import parameters as sp


class IdxOpsUrlNormTestExecutable(sp.ResourceSelector):
    """
    Test binary (built by BUILD_ARCADIA_BINARY task)
    """
    name = 'idx_ops_urlnorm_resource_id'
    description = 'idx_ops urlnorm tests binary'
    resource_type = 'ARCADIA_BINARY_ARCHIVE'
    required = True


class IdxOpsUrlNormTest(SandboxTask):
    """
        Runs idx_ops url normalization test
        and compares it with canonical values
    """
    type = 'RUN_IDX_OPS_URLNORM_TEST'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = (
        IdxOpsUrlNormTestExecutable,
    )

    def on_execute(self):

        def get_test_file(file_name):
            """
            Get test file from SVN by name
            """
            atd = 'arcadia:/arc/trunk/arcadia_tests_data/'
            Arcadia.export(
                '{0}idx_ops/url_normalizer/{1}'.format(atd, file_name),
                self.abs_path(file_name))

        os.chdir(self.abs_path())
        bin_name = 'idx_urlnorm_test'

        bin_pack = bin_name + '-binaries.tar.gz'

        shutil.copy(
            self.sync_resource(self.ctx['idx_ops_urlnorm_resource_id']),
            bin_pack)

        ratings = 'ratings.tsv.gz'
        canonical_gz = 'canonical-errors.log.gz'
        canonical_log = 'canonical-errors.log'
        patches = [
            'canon-patch-0001.patch',
            'canon-patch-0002.patch',
        ]
        output_log = 'output-errors.log'
        output_diff = 'output.diff'

        get_test_file(ratings)
        get_test_file(canonical_gz)
        process.run_process('gunzip ' + canonical_gz, log_prefix='extract-canonical')

        for patch in patches:
            get_test_file(patch)
            process.run_process(
                'patch -p0 ' + canonical_log + ' < ' + patch,
                log_prefix='patch-canonical',
                shell=True,
            )

        process.run_process('tar -xvzf ' + bin_pack, log_prefix='extract-tests')
        process.run_process('chmod +x ' + bin_name, log_prefix='set-exec-rights')
        process.run_process('./' + bin_name, log_prefix='run-test')
        if process.run_process(
                [
                    'diff',
                    '--speed-large-files',
                    canonical_log, output_log,
                    '>', output_diff,
                ],
                log_prefix='compare-results',
                check=False,
                shell=True).returncode != 0:

            self.create_resource(
                'idx_ops urlnorm test diff',
                output_diff,
                resource_types.OTHER_RESOURCE, arch='any'
            )

            eh.check_failed('Compare failed: there is a diff')


__Task__ = IdxOpsUrlNormTest
