# -*- coding: utf-8 -*-

import logging

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts

from sandbox.projects.PSUtil import Revision, read_text, print_folder_tree


class MakeOnly(parameters.SandboxStringParameter):
    name = 'make_only'
    description = 'MAKE_ONLY (путь от корня аркадии, разделитель ";")'
    default_value = 'yweb/webdaemons/iexsrv;yweb/webdaemons/iexsrv/procs/contentline/contentline_sbs;tools/remorphc;' \
                    'dict/gazetteer/compiler/gztcompiler;' \
                    'yweb/webdaemons/iexsrv/procs/contentline/contentline_learner;FactExtract/Parser/tomadictscompiler'


class BuildType(parameters.SandboxStringParameter):
    name = 'make_build_type'
    description = 'MAKE_BUILD_TYPE'
    default_value = 'release'


class Allocator(parameters.SandboxStringParameter):
    name = 'allocator'
    description = 'ALLOCATOR'
    default_value = 'LF'


class ArcadiaPatch(parameters.SandboxStringParameter):
    name = 'arcadia_patch'
    description = 'Apply patch. Input patch text (non-ascii symbols will be ignored) or diff file\'s rbtorrent id ' \
                  '(diff should be taken from arcadia root)'
    multiline = True


class IexBuildBins(ArcadiaTask):
    type = 'IEX_BUILD_BINS'
    execution_space = 5000

    LOCAL_BIN_DIR = 'binaries'
    input_parameters = [Revision, MakeOnly, BuildType, Allocator, ArcadiaPatch]

    client_tags = ctc.Tag.LINUX_PRECISE

    def __init__(self, task_id=0):
        ArcadiaTask.__init__(self, task_id)

    def on_enqueue(self):
        revision = self.ctx['revision']
        binaries_path = self.abs_path('results_dir')
        descr = self.descr + ('. With Patch' if self.ctx['arcadia_patch'] != '' else '')
        attrs = {'revision': revision, 'host': self.host}

        self.create_resource(
            description=descr, resource_path=binaries_path + '/yweb/webdaemons/iexsrv/iexsrv',
            resource_type='IEX', attributes=attrs)
        self.create_resource(
            description=descr,
            resource_path=binaries_path + '/yweb/webdaemons/iexsrv/procs/contentline/contentline_learner/contentline_learner',
            resource_type='IEX_CONTENTLINE_LEARNER', attributes=attrs)
        self.create_resource(
            description=descr,
            resource_path=binaries_path + '/yweb/webdaemons/iexsrv/procs/contentline/contentline_sbs/contentline_sbs',
            resource_type='IEX_CONTENTLINE_SBS', attributes=attrs)
        self.create_resource(
            description=descr, resource_type='IEX_DICTS_COMPILER', attributes=attrs,
            resource_path=binaries_path + '/FactExtract/Parser/tomadictscompiler/tomadictscompiler')
        self.create_resource(
            description=descr, resource_type='IEX_REMORPH_COMPILER', attributes=attrs,
            resource_path=binaries_path + '/tools/remorphc')
        self.create_resource(
            description=descr, resource_type='IEX_GZT_COMPILER', attributes=attrs,
            resource_path=binaries_path + '/dict/gazetteer/compiler/gztcompiler')

    def downloads(self, revision):
        self.ctx[consts.ARCADIA_URL_KEY] = 'arcadia:/arc/trunk/arcadia@' + revision
        return self.get_arcadia_src_dir()

    @staticmethod
    def logging_configs(arcadia_src_dir, revision):
        logging.info('revision: ' + str(revision))
        logging.info(read_text(arcadia_src_dir + '/yweb/webdaemons/iexsrv/CMakeLists.txt'))
        logging.info(read_text(arcadia_src_dir + '/FactExtract/Parser/tomadictscompiler/CMakeLists.txt'))

    def run_build(self, src_dir, results_dir):
        build_type = self.ctx.get('make_build_type')
        build_system = consts.YMAKE_BUILD_SYSTEM
        target_dirs = self.ctx.get('make_only').split(';')
        def_flags = '-DALLOCATOR={}'.format(self.ctx['allocator'])
        sdk.do_build(build_system, src_dir, target_dirs, build_type, clear_build=True,
                          results_dir=results_dir, def_flags=def_flags)

    def apply_patch(self, arcadia_dir):
        if self.ctx['arcadia_patch']:
            Arcadia.apply_patch(arcadia_dir, self.ctx['arcadia_patch'] + '\n', self.log_path())

    def on_execute(self):
        revision = self.ctx['revision']
        self.ctx['arcadia_revision'] = revision
        src_dir = self.downloads(revision)
        self.apply_patch(src_dir)
        IexBuildBins.logging_configs(src_dir, revision)
        binaries_path = self.abs_path('results_dir')
        self.run_build(src_dir, binaries_path)
        print_folder_tree(self, binaries_path, recursive=True)
        self.set_info('build completed')


__Task__ = IexBuildBins
