# -*- coding: utf-8 -*-
from os import listdir, path, system
import re
from time import time

import sandbox.common.types.client as ctc

from sandbox.projects.resource_types import IEX_DICTS_COMPILER, IEX_REMORPH_COMPILER, IEX_GZT_COMPILER, IEX_GRAMMAR_PATCH
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.IexUtil import download_env, get_env_conf
from sandbox.projects.PSUtil import print_folder_tree, read_text, generate_log_str


class TomadictsCompilerResource(LastReleasedResource):
    name = 'tomadicts_compiler_resource_id'
    description = 'tomadictscompiler'
    resource_type = IEX_DICTS_COMPILER


class RemorphCompilerResource(LastReleasedResource):
    name = 'remorph_compiler_resource_id'
    description = 'Remorph compiler'
    resource_type = IEX_REMORPH_COMPILER


class GztCompilerResource(LastReleasedResource):
    name = 'gzt_compiler_resource_id'
    description = 'gzt compiler'
    resource_type = IEX_GZT_COMPILER


class GrammarsConstantPatch(LastReleasedResource):
    name = 'grammars_constant_patch'
    description = 'Grammar patch for reviews'
    resource_type = IEX_GRAMMAR_PATCH


class GrammarsPatch(SandboxStringParameter):
    name = 'grammars_patch'
    description = 'Apply patch to grammars. Input patch text (non-ascii symbols will be ignored) or diff ' \
                  'file\'s rbtorrent id (diff should be taken from arcadia_tests_data/fact_extract/dic)'
    multiline = True


class RemorphPatch(SandboxStringParameter):
    name = 'remorph_patch'
    description = 'Apply patch to remorph. Input patch text (non-ascii symbols will be ignored) or diff ' \
                  'file\'s rbtorrent id (diff should be taken from arcadia_tests_data/fact_extract/geo_in_web or ' \
                  'arcadia_tests_data/fact_extract/mail_events_time)'
    multiline = True


class RemorphGrammarsPatch(SandboxStringParameter):
    name = 'remorph_grammars_patch'
    description = 'Apply patch to remorph. Input patch text (non-ascii symbols will be ignored) or diff ' \
                  'file\'s rbtorrent id (diff should be taken from arcadia_tests_data/yweb/iex/source)'
    multiline = True


class IexBuildGrammars(ArcadiaTask):
    type = 'IEX_BUILD_GRAMMARS'
    execution_space = 12000
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [TomadictsCompilerResource, RemorphCompilerResource, GztCompilerResource, GrammarsConstantPatch,
                        GrammarsPatch, RemorphPatch, RemorphGrammarsPatch]

    def on_enqueue(self):
        compiler = channel.sandbox.get_resource(self.ctx['tomadicts_compiler_resource_id'])
        revision = int(compiler.attributes['revision'])

        self.create_resource(description=self.descr, resource_path='grammars', resource_type='IEX_DICTS',
                             attributes={'revision': revision})

        remorph_path = self.abs_path('remorph')
        self.create_resource(description=self.descr + ' REMORPH', resource_path=remorph_path,
                             resource_type='IEX_REMORPH_GRAMMARS', attributes={'revision': revision})

    def download_remorph_grammars(self, env_conf, revision, remorph_path):
        download_env(self, env_conf, 'remorph', revision)
        if self.ctx['remorph_patch']:
            Arcadia.apply_patch(remorph_path, self.ctx['remorph_patch'] + '\n', self.log_path())
        if self.ctx['remorph_grammars_patch']:
            Arcadia.apply_patch(remorph_path, self.ctx['remorph_grammars_patch'] + '\n', self.log_path())
        run_process(['chmod', '-R', 'a+xwr', remorph_path])
        # change_paths_in_remorph_grammars(self, remorph_path, '/*/*')
        # сделать замену в скрытых файлах
        # change_paths_in_remorph_grammars(self, remorph_path, '/*/.*')

    def download_remorph_env(self, revision):
        # скачать компилятор
        remorph_compiler_path = self.sync_resource(self.ctx['remorph_compiler_resource_id'])
        run_process(['chmod', '-R', 'a+xwr', remorph_compiler_path])

        gzt_compiler_path = self.sync_resource(self.ctx['gzt_compiler_resource_id'])
        run_process(['chmod', '-R', 'a+xwr', gzt_compiler_path])

        # подготовка необходимых каталогов для газетира. Они должны лежать в папке arcadia
        # TODO посмотреть какие папки в точности нужны для газетира и не качать всё подряд
        self.ctx[consts.ARCADIA_URL_KEY] = 'arcadia:/arc/trunk/arcadia@' + str(revision)
        arcadia_src_dir = self.get_arcadia_src_dir()
        run_process(['mkdir', '-p', 'arcadia'])
        run_process(['cp', '-r', '-f', arcadia_src_dir + '/kernel', 'arcadia/kernel'])
        run_process(['cp', '-r', '-f', arcadia_src_dir + '/contrib', 'arcadia/contrib'])
        run_process(['mkdir', '-p', 'arcadia_tests_data/recognize'])
        Arcadia.export('arcadia:/arc/trunk/arcadia_tests_data/recognize', 'arcadia_tests_data/recognize',
                       revision=revision)
        run_process(['mkdir', '-p', 'arcadia_tests_data/wizard/thesaurus'])
        Arcadia.export('arcadia:/arc/trunk/arcadia_tests_data/wizard/thesaurus', 'arcadia_tests_data/wizard/thesaurus',
                       revision=revision)
        return gzt_compiler_path, remorph_compiler_path

    def build_remorph_grammars(self, remorph_path, gzt_compiler_path, remorph_compiler_path, log_info):
        for grammar in listdir(remorph_path):
            start_build = time()
            if not path.exists(remorph_path + '/' + grammar + '/.remorph'):
                continue
            grammar_path = remorph_path + '/' + grammar
            self.set_info('grammar_path = ' + grammar_path)
            # для грамматик, которые не используют каскады base.gzt можно не собирать
            if grammar == 'delivery':
                run_process([gzt_compiler_path, '--force', '-T', self.abs_path(),  remorph_path + '/delivery/base.gzt'],
                            work_dir=grammar_path, log_prefix='base_gzt_build')
            run_process([remorph_compiler_path + '/remorphc', grammar_path + '/.remorph'],
                        work_dir=grammar_path, log_prefix=grammar + '_grammar_build')
            end_build = time()
            duration = int((end_build - start_build)/60)  # min
            log_info += [('build ' + grammar, "{0} min".format(duration), duration > 5)]

    def create_remorph_resource(self, revision, remorph_path):
        self.create_resource(
            description=self.descr + ' REMORPH', resource_path=remorph_path, resource_type='IEX_REMORPH_GRAMMARS',
            attributes={'revision': revision})

    def build_tomita_grammars(self, env_conf):
        # download compiler
        compiler = channel.sandbox.get_resource(self.ctx['tomadicts_compiler_resource_id'])
        compiler_path = self.sync_resource(compiler)
        revision = int(compiler.attributes['revision'])

        # download
        download_env(self, env_conf, 'grammars', revision)
        if self.ctx['grammars_constant_patch'] != 0:
            grammars_constant_patch = read_text(self.sync_resource(self.ctx['grammars_constant_patch']))
            Arcadia.apply_patch(self.abs_path('grammars'), grammars_constant_patch + '\n', self.log_path())
        else:
            self.set_info('Built without constant patch for reviews!')
        if self.ctx['grammars_patch']:
            Arcadia.apply_patch(self.abs_path('grammars'), self.ctx['grammars_patch'] + '\n', self.log_path())

        # run dicts building
        compiler_out_path = self.log_path('out_of_tomadictscompiler.txt')
        cmd = [compiler_path, 'grammars/fdo/aux_dic_kw.cxx grammars grammars/fdo > ' + compiler_out_path + ' 2>&1']
        self.set_info('Run: ' + ' '.join(cmd))
        system(' '.join(cmd))

        # create resource
        self.create_resource(description=self.descr, resource_path='grammars', resource_type='IEX_DICTS',
                             attributes={'revision': revision})

        return revision, compiler_out_path

    def analyze(self, compiler_out_filename):
        compiler_out = read_text(compiler_out_filename)
        cannot_find = (len(re.compile("Cannot find").findall(compiler_out))) > 0
        error = (len(re.compile("FAILED").findall(compiler_out))) > 0
        if cannot_find or error:
            self.set_info('ERROR in grammars build: ' + compiler_out)
        print_folder_tree(self, self.abs_path(), recursive=True)
        return cannot_find or error

    def on_execute(self):
        env_conf = get_env_conf(self)

        # tomita
        revision, compiler_out_path = self.build_tomita_grammars(env_conf)

        # remorph
        remorph_path = self.abs_path('remorph')
        self.download_remorph_grammars(env_conf, revision, remorph_path)
        gzt_compiler_path, remorph_compiler_path = self.download_remorph_env(revision)
        log_info = []
        self.build_remorph_grammars(remorph_path, gzt_compiler_path, remorph_compiler_path, log_info)

        # logging
        system('du -cx grammars | tail -n 1 | grep -o "[0-9]*" > iex_dicts_size')
        iex_dicts_size = int(read_text('iex_dicts_size'))/1000.0
        system('find grammars -name *.bin > bin_files')
        has_bin = len(read_text('bin_files')) > 1
        system('find grammars -name *.cxx > cxx_files')
        has_src = (len(read_text('cxx_files')) > 1)
        has_exception = self.analyze(compiler_out_path)
        log_info += [('exceptions',     has_exception, has_exception),
                     ('has_src',        has_src, not has_src),
                     ('has_bin',        has_bin, not has_bin),
                     ('dicts_size(MB)', iex_dicts_size, iex_dicts_size < 699)]
        if (iex_dicts_size < 699) or (not has_src) or (not has_bin) or has_exception:
            raise SandboxSubprocessError(generate_log_str(log_info))
        self.set_info(generate_log_str(log_info), do_escape=False)


__Task__ = IexBuildGrammars
