# -*- coding: utf-8 -*-

import logging
from os import environ, path, system
import urllib2
import re
import shutil
from time import gmtime, sleep, strftime

import sandbox.common.types.client as ctc

import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
from sandbox.projects.common import apihelpers
from sandbox.projects.resource_types import IEX, IEX_DICTS, IEX_REMORPH_GRAMMARS, MARKET_CONFIG, IEX_CONTENTLINE_MODEL
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process as run
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.IexUtil import download_contentline_model
from sandbox.projects.PSUtil import print_good_info, print_err, yassert, append_text_to_head, has_page, print_folder_tree, \
    read_text, write_text


class IexBuildPackages(SandboxTask):
    type = 'IEX_BUILD_PACKAGES'
    execution_space = 12000
    client_tags = ctc.Tag.Group.LINUX

    class PackagesPatch(parameters.SandboxStringParameter):
        name = 'packages_patch'
        description = (
            'Apply patch. Input patch text (non-ascii symbols will be ignored)'
            ' or diff file\'s rbtorrent id (diff should be taken from arcadia_tests_data/yweb/iex/packages root)'
        )
        multiline = True

    class ChangelogConf(parameters.SandboxStringParameter):
        name = 'changelog_conf'
        description = 'Changelog for conf'

    class ContentlineModelResourceId(parameters.ResourceSelector):
        name = 'contentline_model_resource_id'
        required = True
        description = 'Contentline model'
        resource_type = IEX_CONTENTLINE_MODEL

    class PackConf(parameters.SandboxBoolParameter):
        name = 'pack_conf'
        description = 'Pack conf (don\'t use any resources)'
    PackConf.sub_fields = {'true': [ChangelogConf.name, ContentlineModelResourceId.name]}

    class ChangelogGrammars(parameters.SandboxStringParameter):
        name = 'changelog_grammars'
        description = 'Changelog for grammars'

    class DictsResourceId(parameters.ResourceSelector):
        name = 'dicts_resource_id'
        description = 'Iex dicts'
        resource_type = IEX_DICTS

    class PackGrammars(parameters.SandboxBoolParameter):
        name = 'pack_grammars'
        description = 'Pack grammars (use resource IEX_DICTS)'
    PackGrammars.sub_fields = {'true': [ChangelogGrammars.name, DictsResourceId.name]}

    class ChangelogRemorph(parameters.SandboxStringParameter):
        name = 'changelog_remorph'
        description = 'Changelog for remorph'

    class RemorphResourceId(parameters.ResourceSelector):
        name = 'remorph_resource_id'
        description = 'Remorph grammars (use resource IEX_REMORPH_GRAMMARS)'
        resource_type = IEX_REMORPH_GRAMMARS

    class PackRemorph(parameters.SandboxBoolParameter):
        name = 'pack_remorph'
        description = 'Pack remorph (use resource IEX_REMORPH_GRAMMARS)'
    PackRemorph.sub_fields = {'true': [ChangelogRemorph.name, RemorphResourceId.name]}

    class ChangelogPatterns(parameters.SandboxStringParameter):
        name = 'changelog_patterns'
        description = 'Changelog for patterns'

    class MarketConfigResourceId(parameters.ResourceSelector):
        name = 'market_config_resource_id'
        description = 'Market config'
        resource_type = MARKET_CONFIG

    class PackPatterns(parameters.SandboxBoolParameter):
        name = 'pack_patterns'
        description = 'Pack patterns (use resource MARKET_CONFIG)'
    PackPatterns.sub_fields = {'true': [ChangelogPatterns.name, MarketConfigResourceId.name]}

    class ChangelogServer(parameters.SandboxStringParameter):
        name = 'changelog_server'
        description = 'Changelog for server'

    class IexResourceId(parameters.ResourceSelector):
        name = 'iex_resource_id'
        description = 'Iex binary'
        resource_type = IEX

    class PackServer(parameters.SandboxBoolParameter):
        name = 'pack_server'
        description = 'Pack server (use resource IEX)'
    PackServer.sub_fields = {'true': [ChangelogServer.name, IexResourceId.name]}

    class Upload2Repo(parameters.SandboxBoolParameter):
        name = 'upload_to_repo'
        description = 'Upload to repo'
        default_value = True

    class PackagesAndScriptsSvn(parameters.SandboxStringParameter):
        name = 'packages_and_scripts_svn'
        description = 'Path to svn with packages and scripts'
        default_value = 'arcadia:/arc/trunk/arcadia_tests_data/yweb/iex'

    class Revision(parameters.SandboxStringParameter):
        name = 'revision'
        description = 'Revision'
        default_value = 'HEAD'

    class UseLastChangelog(parameters.SandboxBoolParameter):
        name = 'use_last_changelog'
        description = 'Use last changelog. Do not up version'
        default_value = True

    input_parameters = [
        PackagesPatch,
        PackConf, ChangelogConf, ContentlineModelResourceId,
        PackGrammars, ChangelogGrammars, DictsResourceId,
        PackRemorph, ChangelogRemorph, RemorphResourceId,
        PackPatterns, ChangelogPatterns, MarketConfigResourceId,
        PackServer, ChangelogServer, IexResourceId,
        Upload2Repo, PackagesAndScriptsSvn, Revision, UseLastChangelog
    ]

    DUPLOAD_CONF = {
        'common': {
            'fqdn': "common.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': "robot-iex-ci",
            'incoming': "/repo/common/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        },
        'search': {
            'fqdn': "search.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': "robot-iex-ci",
            'incoming': "/repo/search/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        },
        'yandex-precise': {
            'fqdn': "yandex-precise.dupload.dist.yandex.ru",
            'method': "scpb",
            'login': "robot-iex-ci",
            'incoming': "/repo/yandex-precise/mini-dinstall/incoming/",
            'dinstall_runs': 1,
        }
    }

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self.package_dir = ''

    def yassert(self, expr, msg):
        if not expr:
            raise SandboxSubprocessError('<font="red">ERROR : ' + str(msg) + '</font>')

    # Udalit vozmojnost zagruzki paketa v repozitorij posle togo kak ona v Iex_Release zarabotaet vykatka servera
    def upload_to_repo(self, pack_name):
        platform = 'Unknown'
        if 'precise' in self.client_info['platform']:
            platform = 'precise'
        repo_id = 'yandex-' + platform
        repo_base_url = 'http://' + repo_id + '.dist.yandex.ru'
        repo_url = repo_base_url + '/' + repo_id + '/<dir>/amd64/'
        repo_dirs = ['intra', 'testing', 'prestable', 'unstable', 'stable']
        pack_name_deb = pack_name + '.deb'
        pack_url = repo_url + pack_name_deb
        for folder in repo_dirs:
            if has_page(pack_url.replace('<dir>', folder), self):
                print_err(self, 'ERROR in upload to repo: package ' + pack_url + ' exists')
                return False

        run(['chmod', '-R', 'a+xrw', self.abs_path()])
        with sandbox.projects.common.debpkg.DebRelease(self.DUPLOAD_CONF) as deb:
            with ssh.Key(self, 'IEX-CI', 'robot-iex-ci-ssh-private-key2'):
                work_dir = self.abs_path('_'.join(pack_name.split('_')[:-1]).replace('_', '-'))
                system('cp -f ' + self.abs_path() + '/' + '_'.join(pack_name.split('_')[:-1]) + '* ' + work_dir)
                self.set_info('work_dir = ' + work_dir)
                print_folder_tree(self, self.abs_path(), recursive=True)
                deb.debrelease(['--to', repo_id], work_dir=work_dir)
        sleep(90)

        # test
        packages_url = repo_base_url + '/' + repo_id + '/unstable/amd64/Packages'
        download_url = self.get_package_url(packages_url, repo_id, pack_name_deb)
        if not (has_page(repo_base_url + download_url, self)):
            print_err(self, 'Package ' + pack_name + '.deb did not upload')
            return False
        print_good_info(self, 'Package ' + pack_name + '.deb uploaded')
        return True

    def get_package_url(self, pack_url, env, packname):
        response = urllib2.urlopen(pack_url)
        text = response.read()
        ans = ""
        try:
            regexp = 'Filename: (/storage/\d+/' + env + '/' + packname + ')'
            m = re.search(regexp, text)
            ans = m.group(1)
        except:
            None
        return ans

    def up_version(self, pack_name, new_version, changelog_path):
        if len(self.ctx['changelog_' + pack_name]) < 3:
            print_err(self, 'ERROR with ' + pack_name + ': log too small')
            return False
        log = 'iex-' + pack_name + ' (0.' + str(new_version) + ') unstable; urgency=low\n\n' \
              '  * ' + self.ctx['changelog_' + pack_name] + '\n\n -- IexCi Robot <robot-iex-ci@yandex-team.ru>' \
              '  ' + strftime("%a, %d %b %Y %H:%M:%S +0000", gmtime()) + '\n\n'
        self.set_info(log)
        append_text_to_head(changelog_path, log)

    def debuild(self, name, cmd_args, revision=None):
        pack_dir = path.join(self.package_dir, name)
        debuild_script = self.abs_path('pack-' + name + '.sh')
        Arcadia.export(self.ctx['packages_and_scripts_svn'] + '/scripts/pack-' + name + '.sh', debuild_script,
                       self.get_revision())
        run(['chmod', 'a+rwx', debuild_script])

        # podlojit changelog iz resursa
        last_pack = apihelpers.get_last_resource_with_attrs(
            resource_type='IEX_DEB', attrs={'package': name}, all_attrs=True)
        if last_pack is None:
            self.set_info('ERROR in debuild ' + name + ': last resource IEX_DEB not found.')
            return False
        last_changelog_path = self.sync_resource(last_pack) + '/debian/changelog'

        changelog_path = path.join(pack_dir, 'changelog')
        write_text(changelog_path, read_text(last_changelog_path))

        environment = environ.copy()
        environment['DEBEMAIL'] = 'IexCi Robot <robot-iex-ci@yandex-team.ru>'

        with sandbox.projects.common.gnupg.GpgKey(self, 'IEX-CI', 'robot-iex-ci-gpg-private', 'robot-iex-ci-gpg-public'):
            new_version = int(last_pack.attributes['version'])
            if not self.ctx['use_last_changelog']:
                new_version += 1
                self.up_version(name, new_version, changelog_path)
            self.set_info('Package version: ' + str(new_version))

            # zapustit  debuild
            run(['gpg', '--list-secret-keys'], log_prefix='gpg_list-secret-keys')
            run(['chmod', 'a+x', path.join(pack_dir, 'rules')])
            print_folder_tree(self, self.abs_path(), recursive=True)
            cmd = [debuild_script, '-s', self.package_dir, '-r'] + cmd_args
            self.set_info('Run ' + ' '.join(cmd))
            debuild_log_filename = 'pack_' + name
            run(cmd, log_prefix=debuild_log_filename)
            debuild_log = read_text(self.log_path(debuild_log_filename + '.out.txt'))
            if 'fatal error' in debuild_log:
                self.set_info('ERROR in debuild: ' + name + ' ne byl sobran:\n' + debuild_log)
                return False

        success = True

        # upload to repo
        if self.ctx['upload_to_repo']:
            success = self.upload_to_repo('iex-' + name + '_0.' + str(new_version) + '_amd64')

        # sozdat resurs s paketom
        pack_path = self.abs_path('iex-' + name + '-0.' + str(new_version))
        run(['chmod', '-R', 'a+xrw', self.abs_path()])
        system('cp -fR ' + self.abs_path('iex-' + name + '_0.' + str(new_version)) + '* ' + pack_path)
        self.create_resource(
            description=self.descr, resource_path=pack_path, resource_type='IEX_DEB',
            attributes={'revision': str(revision), 'package': name, 'released': 'False', 'ttl': '10',
                        'backup_task': True, 'version': new_version})

        return success

    def get_revision(self):
        if self.ctx['revision'] == 'HEAD':
            return None
        elif self.ctx['revision'].isdigit():
            return self.ctx['revision']
        yassert(False, 'Neverno ukazana reviziya.')

    def on_execute(self):
        import sys
        reload(sys)
        sys.setdefaultencoding('utf-8')

        platform = self.client_info['platform']
        self.set_info('Platform: ' + platform)
        success = True

        # package dir
        self.package_dir = self.abs_path('packages')
        make_folder(self.package_dir)
        Arcadia.export(self.ctx['packages_and_scripts_svn'] + '/packages/', self.package_dir, self.get_revision())
        if self.ctx['packages_patch'] != '':
            Arcadia.apply_patch(self.package_dir, self.ctx['packages_patch'] + '\n', self.log_path())
        run(['chmod', '-R', 'a+xrw', self.package_dir])
        logging.info('Package dir: ' + self.package_dir)
        Arcadia.export(self.ctx['packages_and_scripts_svn'] + '/scripts/utils.sh', self.abs_path('utils.sh'),
                       self.get_revision())

        # debuild packs
        yassert(self.ctx['pack_conf'] or self.ctx['pack_grammars'] or self.ctx['pack_remorph']
                or self.ctx['pack_patterns'] or self.ctx['pack_server'], 'Ne vybran ni odin paket!')

        if self.ctx['pack_conf']:
            download_contentline_model(self, self.abs_path('packages/settings'))
            Arcadia.export('arcadia:/arc/trunk/arcadia_tests_data/recognize/dict.dict', self.package_dir + '/dict.dict',
                           self.get_revision())
            success = success and (self.debuild('conf', []))

        if self.ctx['pack_grammars']:
            dicts_path = self.sync_resource(self.ctx['dicts_resource_id'])
            grammars_dir = self.abs_path('grammars')
            make_folder(grammars_dir)
            run(['cp', '-r', dicts_path, self.abs_path()])
            run(['chmod', '-R', 'a+w', grammars_dir])
            success = success and (self.debuild('grammars', ['-g', grammars_dir]))

        if self.ctx['pack_remorph']:
            remorph_path = self.sync_resource(self.ctx['remorph_resource_id'])
            remorph_dir = self.abs_path('remorph')
            make_folder(remorph_dir)
            system('cp -r ' + remorph_path + '/* ' + remorph_dir)
            run(['chmod', '-R', 'a+w', remorph_dir])
            success = success and (self.debuild('remorph', ['-g', remorph_dir]))

        if self.ctx['pack_patterns']:
            market_config_path = self.sync_resource(self.ctx['market_config_resource_id'])
            run(['cp', market_config_path, self.package_dir], log_prefix='mv_config')
            success = success and self.debuild('patterns', [])

        if self.ctx['pack_server']:
            iex_path = self.sync_resource(self.ctx['iex_resource_id'])
            revision = channel.sandbox.get_resource(self.ctx['iex_resource_id']).attributes['revision']
            iex_dir = self.abs_path('build/yweb/webdaemons/iexsrv')
            make_folder(iex_dir)
            shutil.copy(iex_path, iex_dir)
            success = success and (self.debuild('server', ['-b', iex_dir], revision))

        yassert(success, '')


__Task__ = IexBuildPackages
