#!/usr/bin/env python
# -*- coding: utf-8 -*-
import json
import os
import re

import sandbox.common.types.client as ctc

from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.IexUtil import ENTITIES, DOCS_EXT
from sandbox.projects.PSUtil import generate_log_str, map_of_map_to_html_table, create_file_with_content, read_text


class Entities(SandboxStringParameter):
    name = 'entities'
    description = 'Перечислите через запятую без пробелов сущности для которых вы хотите увидеть результаты. ' \
                  'Для каждой сущности скачаются последние создные в SB ресурсы c результатами'
    default_value = ','.join(ENTITIES)


class IexGetLastResults(task.SandboxTask):
    type = 'IEX_GET_LAST_RESULTS'
    execution_space = 10000
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [Entities]

    def __init__(self, task_id=0):
        task.SandboxTask.__init__(self, task_id)
        self.log_info = {}

    def yassert(self, expr, msg):
        if not expr:
            self.panic(msg)

    def panic(self, msg=''):
        log = []
        if msg != '':
            log = [('Task in ERROR state because', msg, True)]
        raise SandboxSubprocessError(map_of_map_to_html_table(self.log_info) + '<br>' + generate_log_str(log))

    @staticmethod
    def get_diffs_url(entity):
        diffs = apihelpers.get_last_resource_with_attrs('IEX_DIFF_FUNC_RESULTS', attrs={'entity': entity})
        if diffs:
            return diffs.url
        return None

    def download_tests(self, entity):
        r = apihelpers.get_last_resource_with_attrs('IEX_TESTS', all_attrs=True, attrs={entity: 'yes'})
        self.yassert(r, 'Тесты не найдены, entity=' + entity)
        return self.sync_resource(r), r.url

    def download_tests_results(self, entity):
        r = apihelpers.get_last_resource_with_attrs('IEX_FUNC_TEST_RESULT', all_attrs=True, attrs={'entity': entity})
        self.yassert(r, 'Результаты тестирования не найдены, entity=' + entity)
        return self.sync_resource(r)

    @staticmethod
    def get_info(entity, test_info, tests_results, tests_url):
        query = test_info['query']
        domain = 'None'
        has_domain = re.search('domain=([^&]*)', query)
        if has_domain:
            domain = has_domain.group(1)
        label = 'None'
        if ('labels' in test_info) and (','.join(test_info['labels'])):
            label = ','.join(test_info['labels'])
        id_str = str(test_info['id'])
        ans_path = tests_results + '/' + entity + '/' + id_str + '.txt'
        ans = 'No such file: ' + ans_path
        if os.path.exists(ans_path):
            ans = read_text(ans_path)
        doc_url = tests_url + '&relpath=/doc/' + id_str + '.' + DOCS_EXT[entity]
        return domain, label, id_str, ans, doc_url

    @staticmethod
    def get_data(entity, tests, tests_url, tests_results):
        data = {}
        tests_info = json.loads(open(tests + '/tests_data.json').read())
        for test_info in tests_info:
            domain, label, id_str, ans, doc_url = IexGetLastResults.get_info(entity, test_info, tests_results, tests_url)
            if label not in data:
                data[label] = {}
            data_by_label_and_domain = data[label][domain][1] + '<br>' if domain in data[label] else ''
            data[label][domain] = \
                ['', data_by_label_and_domain + '<a href=' + doc_url + '>' + id_str + '</a> <samp>' + ans + '</samp>',
                 False]
        return data

    @staticmethod
    def get_data_with_docs(entity, tests, tests_url, tests_results):
        data = {}
        tests_info = json.loads(open(tests + '/tests_data.json').read())
        for test_info in tests_info:
            domain, label, id_str, ans, doc_url = IexGetLastResults.get_info(entity, test_info, tests_results, tests_url)
            if label not in data:
                data[label] = {}
            data_by_label_and_domain = data[label][domain][1] if domain in data[label] else ''
            dates = re.findall('("date_dep":"[^"]*")', ans) + re.findall('("date_arr":"[^"]*")', ans)
            dates += re.findall('("date_dep_back":"[^"]*")', ans) + re.findall('("date_arr_back":"[^"]*")', ans)
            dates = '<br>'.join(dates)
            frame = '<iframe src="' + doc_url + '" width="800" height="1000" align="left"></iframe>'
            data[label][domain] = \
                ['', data_by_label_and_domain + '<table><tr><td>' + frame + '<a href=' + doc_url + '>' + id_str +
                 '</a> <samp>' + ans + '</samp><br><b>' + dates + '</b></td></tr></table>', False]
        return data

    def save_map_as_html(self, entity, data, result):
        create_file_with_content(result, map_of_map_to_html_table(data, align='left'))
        r = self.create_resource(description=self.descr, resource_type='IEX_RESULTS', resource_path=result,
                                 attributes={'ttl': 100, 'entity': entity})
        return r.url

    def get_last_results(self, entity):
        # добавить сущность в лог
        self.log_info[entity] = {}

        # диффы
        diffs_url = IexGetLastResults.get_diffs_url(entity)
        self.log_info[entity]['diffs'] = [diffs_url, '☼', not diffs_url]

        # скачать последние тесты и последние результаты тестов
        tests, tests_url = self.download_tests(entity)
        tests_results = self.download_tests_results(entity)

        # по тестам и результатам тестов построить map с данными
        data = IexGetLastResults.get_data(entity, tests, tests_url, tests_results)
        # сохранить map как html
        result_url = self.save_map_as_html(entity, data, entity + '_res.html')
        self.log_info[entity]['briefly'] = [result_url, '☼', False]

        # по тестам и результатам тестов построить map с данными с документами
        spec_data = IexGetLastResults.get_data_with_docs(entity, tests, tests_url, tests_results)
        # сохранить map как html
        spec_result_url = self.save_map_as_html(entity, spec_data, entity + '_spec_res.html')
        self.log_info[entity]['with docs'] = [spec_result_url, '☼', not spec_result_url]

    def on_execute(self):
        entities = self.ctx['entities'].split(',')
        for entity in entities:
            self.get_last_results(entity)
        self.set_info(map_of_map_to_html_table(self.log_info), do_escape=False)


__Task__ = IexGetLastResults
