#!/usr/bin/env python
# -*- coding: utf-8 -*-
import json
from multiprocessing import Process
from os import path, system

import sandbox.common.types.client as ctc

from sandbox.projects.common.FuncTestSuiteBase import FuncTestSuiteBase
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.IexUtil import TESTS, DOCS_EXT, FIELDS_FOR_DOWNLOAD, download_iex
from sandbox.projects.PSUtil import create_file_with_content, append_text, generate_log_str, get_port, \
    check_that_server_running


class IexTestFunc(FuncTestSuiteBase):
    type = 'IEX_TEST_FUNC'
    execution_space = 10000
    environment = (SvnEnvironment(), )
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = FIELDS_FOR_DOWNLOAD + TESTS

    def __init__(self, task_id=0):
        FuncTestSuiteBase.__init__(self, task_id)
        self.log_info = []

    def yassert(self, expr, msg=''):
        if not expr:
            self.panic(msg)

    def panic(self, msg=''):
        if msg != '':
            self.log_info += [('Task in ERROR state because', msg, True)]
        raise SandboxSubprocessError(generate_log_str(self.log_info, '<br>'))

    def run_test(self, e, port, answs_dir, data):
        data['tests_dir'] = self.sync_resource(self.ctx[e + '_tests_resource_id'])
        data['tests_info'] = json.loads(open(data['tests_dir'] + '/tests_data.json').read())
        data['core_path'] = self.log_path(e + '.exception.txt')

        # test
        p = Process(target=IexTestFunc.test, args=(e, data, port, answs_dir + '/' + e, ))
        data['process'] = p
        p.start()

    @staticmethod
    def send_requests(entity, port, data, answs_dir):
        make_folder(answs_dir)
        for test_info in data['tests_info']:
            doc_path = data['tests_dir'] + '/doc/' + str(test_info['id']) + '.' + DOCS_EXT[entity]
            ans_path = answs_dir + '/' + str(test_info['id']) + '.txt'
            if path.exists(doc_path):
                wget_arg = test_info['wget_arg'] if ('wget_arg' in test_info) and (test_info['wget_arg']) else ''
                system("wget --max-redirect 0 -t 1 -T 5 -O '" + ans_path + "' " + wget_arg + " 'localhost:" + str(port)
                       + test_info['query'] + "' --post-file='" + doc_path + "'")
            else:
                append_text(data['core_path'], 'ERROR: Document ' + doc_path + ' has no exists\n')

    @staticmethod
    def test(entity, data, port, answs_dir):
        try:
            IexTestFunc.send_requests(entity, port, data, answs_dir)
        except Exception as e:
            create_file_with_content(data['core_path'], str(e.message) + '\n\n' + str(e.args))

    def on_execute(self):
        # download data
        _, server_path, config_path = download_iex(self, config_id=self.ctx['config_id'])
        port = get_port(config_path)
        self.set_info('iex downloaded')

        # run iex
        run_process([server_path, config_path], log_prefix='run_iex', wait=False, work_dir=self.log_path())

        # check run
        sec = check_that_server_running(self, port=port)
        self.log_info += [('start time', sec, sec > 40)]

        if self.ctx['tested_entities'] == '':
            self.set_info(generate_log_str(self.log_info, '<br>'), do_escape=False)
            return

        answs_dir = self.abs_path('answers')
        self.create_resource(description=self.descr, resource_type='IEX_FUNC_TEST_RESULT', resource_path=answs_dir,
                             attributes={'entity': self.ctx['tested_entities']})

        data = {}
        self.set_info('run tests')
        for entity in self.ctx['tested_entities'].split(','):
            data[entity] = {}
            self.run_test(entity, port, answs_dir, data[entity])

        self.set_info('wait tests')
        success = True
        for entity in data:
            e_data = data[entity]
            e_data['process'].join()
            if path.exists(e_data['core_path']):
                success = False
                continue

        self.yassert(success, 'Has exception. See log for details.')
        check_that_server_running(self, port=port)
        self.set_info(generate_log_str(self.log_info, '<br>'), do_escape=False)


__Task__ = IexTestFunc
