# -*- coding: utf-8 -*-

from base64 import b64decode
import json
from os import system
from urllib2 import HTTPError

import sandbox.common.types.client as ctc

from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter, LastReleasedResource
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process

from sandbox.projects.IexUtil import ENTITIES, DOCS_EXT, ENTITIES_WITH_TESTS_IN_AQUA
from sandbox.projects.PSUtil import download_text, download_and_get_text, get_md5_hash, map_of_map_to_html_table, create_file, \
    create_file_with_content, print_folder_tree


class Entities(SandboxStringParameter):
    name = 'entities'
    description = 'Entities (separator is ,)'
    default_value = ','.join(ENTITIES_WITH_TESTS_IN_AQUA)


class TestsUrl(SandboxStringParameter):
    name = 'pattern_tests_url'
    description = 'Tests url'
    default_value = 'http://aqua.yandex-team.ru/storage/get/ru/yandex/autotests/innerpochta/tomita/core/' \
                    'ContentGenUtils/data-export-<entity>.txt'


class DoUpdate(SandboxBoolParameter):
    name = 'do_update'
    description = 'Do update prod tests'
    sub_fields = {
        'true': [Entities.name, TestsUrl.name]
    }


class Entity(SandboxStringParameter):
    name = 'entity'
    description = 'Сущность для которой нужно обновить тесты'
    choices = tuple([(entity, entity) for entity in ENTITIES + ['sanitizer']])


class WgetArg(SandboxStringParameter):
    name = 'wget_arg'
    description = 'Add arg to wget'


class Test1ForAdd(SandboxStringParameter):
    name = 'test_for_add1'
    description = 'Текст теста, который требуется добавить'
    multiline = True


class Test2ForAdd(SandboxStringParameter):
    name = 'test_for_add2'
    description = 'Текст теста, который требуется добавить'
    multiline = True


class Test3ForAdd(SandboxStringParameter):
    name = 'test_for_add3'
    description = 'Текст теста, который требуется добавить'
    multiline = True


class Test4ForAdd(SandboxStringParameter):
    name = 'test_for_add4'
    description = 'Текст теста, который требуется добавить'
    multiline = True


class Test5ForAdd(SandboxStringParameter):
    name = 'test_for_add5'
    description = 'Текст теста, который требуется добавить'
    multiline = True


class Query(SandboxStringParameter):
    name = 'query'
    description = 'Текст запроса'
    default_value = '/?e=addr'


class Labels(SandboxStringParameter):
    name = 'labels'
    description = 'Метка. Для классификатора можно указать меток больше одной метки через запятую без пробелов'


class IsBase64(SandboxBoolParameter):
    name = 'is_base64'
    description = 'Документы в base64'
    default_value = False


class UseUnixEOL(SandboxBoolParameter):
    name = 'use_unix_eol'
    description = 'Заменить \\r\\n и \\r на \\n'
    default_value = True


class CreateNewResource (SandboxBoolParameter):
    name = 'create_new_resource'
    description = 'Create new resource if has no someone else'
    default_value = False


class DoAdd(SandboxBoolParameter):
    name = 'do_add'
    description = 'Do add tests'
    sub_fields = {
        'true': [Entity.name, WgetArg.name, Query.name, Labels.name, IsBase64.name, UseUnixEOL.name,
                 CreateNewResource.name, Test1ForAdd.name, Test2ForAdd.name, Test3ForAdd.name, Test4ForAdd.name,
                 Test5ForAdd.name]
    }


class TestsResourceId(LastReleasedResource):
    name = 'tests_resource_id'
    description = 'tests_resource_id'
    resource_type = ['IEX_TESTS']


class Entity2(SandboxStringParameter):
    name = 'entity2'
    description = 'Сущность для которой нужно распаковать тесты'
    choices = tuple([(entity, entity) for entity in ENTITIES])


class DoUnpackTar(SandboxBoolParameter):
    name = 'do_unpack_tar_with_tests'
    description = 'do_unpack_tar_with_tests'
    sub_fields = {
        'true': [TestsResourceId.name, Entity2.name]
    }


class TestsForDeleting(SandboxStringParameter):
    name = 'tests_for_deleting'
    description = 'Тесты, которые нужно удалить без пробелов через запятую'


class Entity3(SandboxStringParameter):
    name = 'entity3'
    description = 'Сущность для которой нужно удалить тесты'
    choices = tuple([(entity, entity) for entity in ENTITIES])


class DoDeleteTests(SandboxBoolParameter):
    name = 'do_delete_tests'
    description = 'Удалить тесты'
    sub_fields = {
        'true': [TestsForDeleting.name, Entity3.name]
    }


class DoSmth(SandboxBoolParameter):
    name = 'do_smth'
    description = 'Do smth'


class IexTestsManager(SandboxTask):
    type = 'IEX_TESTS_MANAGER'

    execution_space = 5000
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [
        DoUpdate, Entities, TestsUrl,
        DoAdd, Entity, WgetArg, Query,  Labels, IsBase64, UseUnixEOL, CreateNewResource, Test1ForAdd, Test2ForAdd,
        Test3ForAdd, Test4ForAdd, Test5ForAdd,
        DoUnpackTar, TestsResourceId, Entity2,
        DoDeleteTests, TestsForDeleting, Entity3,
        DoSmth
    ]

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)
        self.log_info = {}
        self.success = True

    # common

    def download_current_tests(self, entity):
        tests_resource = apihelpers.get_last_resource_with_attrs('IEX_TESTS', attrs={entity: 'yes'})
        cur_tests_dir = self.abs_path('tests/' + entity)
        make_folder(cur_tests_dir)
        if tests_resource is None:
            if self.ctx['create_new_resource']:
                make_folder(cur_tests_dir + '/doc')
                create_file_with_content(cur_tests_dir + '/tests_data.json', '[]')
                info = json.loads(open(cur_tests_dir + '/tests_data.json').read().decode("utf-8-sig"))
                return cur_tests_dir, info
            if entity not in self.log_info:
                self.log_info[entity] = {}
            if 'errors' not in self.log_info[entity]:
                self.log_info[entity]['errors'] = ['', [], False]
            self.log_info[entity]['errors'][1] += ['No resource IEX_TESTS']
            return None, None
        folder = self.sync_resource(tests_resource)
        system('cp -r ' + folder + '/* ' + cur_tests_dir)
        process.run_process(['chmod', '-R', 'a+w', cur_tests_dir])
        print_folder_tree(self, self.abs_path(), recursive=True)
        info = json.loads(open(cur_tests_dir + '/tests_data.json').read().decode("utf-8-sig"))
        return cur_tests_dir, info

    @staticmethod
    def get_max_id(cur_tests_info):
        max_id = 0
        for cur_test_info in cur_tests_info:
            if max_id < cur_test_info['id']:
                max_id = cur_test_info['id']
        return max_id

    def add_test(self, e, (cur_tests_info, doc_dir), (query, new_test_doc_path, labels, wget_arg)):
        next_id = IexTestsManager.get_max_id(cur_tests_info) + 1
        prod_doc_md5 = get_md5_hash(new_test_doc_path)
        self.log_info[e]['added tests'][1] += 1
        new_test = {'id': next_id, 'query': query, 'doc_hash': prod_doc_md5, 'labels': labels, 'wget_arg': wget_arg}
        extended_docs_ext = DOCS_EXT
        extended_docs_ext['sanitizer'] = 'html'
        process.run_process(['cp', new_test_doc_path, doc_dir + '/' + str(next_id) + '.' + extended_docs_ext[e]])
        return new_test

    @staticmethod
    def update_cur_tests_info(cur_tests_info, cur_tests_dir):
        process.run_process(['rm', cur_tests_dir + '/tests_data.json'])
        tests_data_file = open(cur_tests_dir + '/tests_data.json', 'w')
        json.dump(cur_tests_info, tests_data_file, indent=4, sort_keys=True, ensure_ascii=False)

    # update

    def read_test_data(self, test_data):
        test_params = test_data.split('::')
        if len(test_params) != 3:
            self.set_info('WARNING: has no all info for test. test_data=' + str(test_params))
            return None
        query, doc_path, canonic_path = test_params
        if len(canonic_path) < 3:
            self.set_info('WARNING: canonic_path very strange. canonic_path=' + str(canonic_path))
            return None
        return query, doc_path, canonic_path

    def download_prod_tests(self, entity, tests_dir):
        tests_url = self.ctx['pattern_tests_url'].replace('<entity>', entity)
        try:
            tests_data_str = download_and_get_text(tests_url)
            if len(tests_data_str) == 0:
                self.log_info[entity]['errors'][1] += ['No tests from url: ' + tests_url]
                return None
        except HTTPError:
            self.log_info[entity]['errors'][1] += [
                'Cannot download tests from url: ' + tests_url + '. Update tests in aqua: https://wiki.yandex-team.ru/ps/iex/tech']
            return None

        make_folder(tests_dir + '/doc')
        tests_data = []
        for i, test_data in enumerate(tests_data_str.splitlines()):
            data = self.read_test_data(test_data)
            if data is None:
                self.log_info[entity]['incorrect tests'][1] += 1
                continue
            query, doc_url, _ = data

            # download test doc
            DOCS_EXT['sanitizer'] = 'html'
            doc_path = tests_dir + '/doc/' + str(i) + '.' + DOCS_EXT[entity]
            if not download_text(doc_url, doc_path):
                self.log_info[entity]['errors'][1] += ['Cannot download prod doc: ' + doc_url]
                continue
            tests_data += [(query, doc_path)]
        return tests_data, len(tests_data_str.splitlines())

    @staticmethod
    def find_same_test((query, prod_doc_md5), cur_tests_info):
        for cur_test_info in cur_tests_info:
            if (cur_test_info['doc_hash'] == prod_doc_md5) and (cur_test_info['query'] == query):
                return cur_test_info
        return None

    def update_by_prod_tests(self, entity, cur_tests_info, cur_tests_dir, prod_tests_info):
        DOCS_EXT['sanitizer'] = 'html'
        doc_dir = cur_tests_dir + '/doc'

        for query, prod_doc_path in prod_tests_info:
            cur_test_info = IexTestsManager.find_same_test((query, get_md5_hash(prod_doc_path)), cur_tests_info)
            if cur_test_info is None:
                new_test = self.add_test(entity, (cur_tests_info, doc_dir), (query, prod_doc_path, [], None))
                cur_tests_info += [new_test]
            else:
                self.log_info[entity]['seen tests'][1] += 1
                process.run_process(['mv', prod_doc_path, doc_dir + '/' + str(cur_test_info['id']) + '.' + DOCS_EXT[entity]])
        IexTestsManager.update_cur_tests_info(cur_tests_info, cur_tests_dir)

    def do_update(self):
        for e in self.ctx['entities'].split(','):
            self.set_info('process ' + e)
            self.log_info[e] = {}
            e_info = self.log_info[e]
            e_info['added tests'] = ['', 0, False]
            e_info['seen tests'] = ['', 0, False]
            e_info['errors'] = ['', [], False]
            e_info['incorrect tests'] = ['', 0, False]

            # download current tests
            cur_tests_dir, cur_tests_info = self.download_current_tests(e)
            if (cur_tests_dir is None) or (cur_tests_info is None):
                continue

            # download prod tests
            prod_tests_dir = self.abs_path(e + '_prod_tests')
            make_folder(prod_tests_dir)
            prod_info = self.download_prod_tests(e, prod_tests_dir)
            if prod_info is None:
                continue
            prod_tests_info, prod_tests_count = prod_info

            self.update_by_prod_tests(e, cur_tests_info, cur_tests_dir, prod_tests_info)
            r = None
            if e_info['added tests'][1] > 0:
                r = self.create_resource(resource_path=cur_tests_dir, resource_type='IEX_TESTS',
                                         description=e + ' tests', attributes={e: 'yes', 'ttl': 'inf'})

            if e_info['incorrect tests'][1] > 0:
                e_info['incorrect tests'][2] = True
                if e_info['incorrect tests'][1] > prod_tests_count/10:
                    e_info['errors'][1] += ['Many incorrect tests']

            if e_info['added tests'][1] + e_info['seen tests'][1] + e_info['incorrect tests'][1] != prod_tests_count:
                e_info['errors'][1] += ['added+seen+incorrect != prod_tests_count']
                self.set_info('prod_tests_count=' + str(prod_tests_count))

            if (r is not None) and (len(e_info['errors'][1]) == 0) and (e_info['incorrect tests'][1] == 0):
                self.mark_resource_ready(r)

        for e in self.ctx['entities'].split(','):
            e_info = self.log_info[e]
            if len(e_info['errors'][1]) != 0:
                e_info['errors'][2] = True
                self.success = False

    def do_add(self):
        e = self.ctx['entity']

        self.log_info[e] = {}
        self.log_info[e]['errors'] = ['', [], False]
        self.log_info[e]['added tests'] = ['', 0, False]

        # download current tests
        cur_tests_dir, cur_tests_info = self.download_current_tests(e)
        if (cur_tests_dir is None) or (cur_tests_info is None):
            self.success = False
            return

        for i in range(1, 6):
            if self.ctx['test_for_add' + str(i)] == '':
                continue
            new_test_text = self.ctx['test_for_add' + str(i)]
            if self.ctx['is_base64']:
                t = b64decode(new_test_text)
                new_test_text = t
            if self.ctx['use_unix_eol']:
                new_test_text = new_test_text.replace('\r\n', '\n').replace('\r', '\n')

            new_test_doc_path = 'new_test_doc' + str(i)
            create_file_with_content(new_test_doc_path, new_test_text)
            new_test_ans_path = 'new_test_canonic' + str(i)
            create_file(new_test_ans_path)

            new_test = self.add_test(
                e, (cur_tests_info, cur_tests_dir + '/doc'),
                (self.ctx['query'], new_test_doc_path, self.ctx['labels'].split(','), self.ctx['wget_arg']))
            cur_tests_info += [new_test]

        IexTestsManager.update_cur_tests_info(cur_tests_info, cur_tests_dir)
        self.create_resource(description='Add new tests. ' + e, resource_path=cur_tests_dir, resource_type='IEX_TESTS',
                             attributes={e: 'yes', 'ttl': 'inf', 'backup_task': True})

    def do_unpack_tar_with_tests(self):
        p = self.sync_resource(self.ctx['tests_resource_id'])
        process.run_process(['cp', p, self.abs_path()])
        process.run_process(['tar', '-xvf', p.split('/')[-1]])
        e = self.ctx['entity2']
        self.create_resource(
            description='only docs for ' + e, resource_path=self.abs_path(e), resource_type='IEX_TESTS',
            attributes={e: 'yes'})

    def do_delete_tests(self):
        e = self.ctx['entity3']
        cur_tests_dir, cur_tests_info = self.download_current_tests(e)
        if (cur_tests_dir is None) or (cur_tests_info is None):
            self.success = False
            return
        tests_for_deleting = [int(n) for n in self.ctx['tests_for_deleting'].split(',')]
        for test in tests_for_deleting:
            # удалить запись о тесте с данным номером
            for i, cur_test_info in enumerate(cur_tests_info):
                if test == cur_test_info['id']:
                    del cur_tests_info[i]
                    continue

            # удалить документ
            DOCS_EXT['sanitizer'] = 'html'
            system('rm -rf ' + cur_tests_dir + '/doc/' + str(test) + '.' + DOCS_EXT[e])

        # перезаписать файл с данными о тестах
        IexTestsManager.update_cur_tests_info(cur_tests_info, cur_tests_dir)

        # создать ресурс
        self.create_resource(description='Delete tests. ' + e, resource_path=cur_tests_dir, resource_type='IEX_TESTS',
                             attributes={e: 'yes', 'ttl': 'inf', 'backup_task': True})

    def do_smth(self):
        for e in ENTITIES:
            r = apihelpers.get_last_resource_with_attrs('IEX_TESTS', attrs={e: 'yes'}, all_attrs=True)
            p = self.sync_resource(r)
            make_folder(e)
            system('cp -R ' + p + '/doc ' + self.abs_path(e))
            system('cp -R ' + p + '/tests_data.json ' + self.abs_path(e))
            system('chmod  a+xrw -R ' + self.abs_path(e))
            print_folder_tree(self, self.abs_path(), recursive=True)
            self.create_resource(
                description='only docs for ' + e, resource_path=self.abs_path(e), resource_type='IEX_TESTS',
                attributes={e: 'yes'})

    def on_execute(self):
        if self.ctx['do_update']:
            self.do_update()
        elif self.ctx['do_add']:
            self.do_add()
        elif self.ctx['do_unpack_tar_with_tests']:
            self.do_unpack_tar_with_tests()
        elif self.ctx['do_delete_tests']:
            self.do_delete_tests()
        elif self.ctx['do_smth']:
            self.do_smth()
        else:
            self.set_info('Действие не было выбрано')

        if not self.success:
            raise SandboxSubprocessError(map_of_map_to_html_table(self.log_info))
        self.set_info(map_of_map_to_html_table(self.log_info), do_escape=False)


__Task__ = IexTestsManager
