# -*- coding: utf-8 -*-
import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.svn import ArcadiaTestData
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.IexUtil import download_contentline_learner, download_contentline_model
from sandbox.projects.PSUtil import download_archive_and_extract, print_folder_tree


class ContentlineLearner(LastReleasedResource):
    name = 'iex_contentline_learner_resource_id'
    description = 'Бинарник contentline_learner'
    resource_type = ['IEX_CONTENTLINE_LEARNER']


class TrainingSetUrl(SandboxStringParameter):
    name = 'training_set_url'
    description = 'Url до архива в tr.gz c обучающей выборкой'


class ContentlineModel(LastReleasedResource):
    name = 'contentline_model_resource_id'
    description = 'Модель для contentline'
    resource_type = ['IEX_CONTENTLINE_MODEL']


class TestSetUrl(SandboxStringParameter):
    name = 'testing_set_url'
    description = 'Url до архива в tr.gz c тестовой выборкой'


class TrainingSetToModelAndTestSetToSbs(SandboxBoolParameter):
    name = 'training_set_to_model_and_test_set_to_sbs'
    description = 'По обучающей выборке построить модель, по тестовой выборке (если есть) построить данные для SbS'
    sub_fields = {
        'true': [TrainingSetUrl.name]
    }


class ModelAndTestSetToSbs(SandboxBoolParameter):
    name = 'model_and_test_set_to_sbs'
    description = 'По модели и тестовой выборке построить данные для SbS'
    sub_fields = {
        'true': [ContentlineModel.name]
    }


class IexUpdateContentlineModel(SandboxTask):
    type = 'IEX_UPDATE_CONTENTLINE_MODEL'
    client_tags = ctc.Tag.LINUX_PRECISE
    input_parameters = [ContentlineLearner, TestSetUrl, TrainingSetToModelAndTestSetToSbs, TrainingSetUrl,
                        ModelAndTestSetToSbs, ContentlineModel]

    def yassert(self, expr, msg):
        if not expr:
            raise SandboxSubprocessError('<font="red">ERROR : ' + str(msg) + '</font>')

    def generate_sbs_data(self, contentline_learner_path, greetings_path, model_path):
        self.yassert(self.ctx['testing_set_url'] != '', 'Укажите url тестовой выборки!')

        # скачать тестовую выборку
        testing_set_path = self.abs_path('testing_set')
        download_archive_and_extract(self.ctx['testing_set_url'], testing_set_path)

        # запустить contentline learner на тестирующей выборке
        data_for_sbs = self.abs_path('data_for_sbs.txt')
        run_process([contentline_learner_path, '--apply', '--folder', testing_set_path, '--greetings', greetings_path,
                     '--model', model_path], stdout=open(data_for_sbs, 'w'))

        # создать ресурс с данными для SbS
        self.create_resource(description=self.descr, resource_path=data_for_sbs,
                             resource_type='IEX_CONTENTLINE_DATA4SBS',
                             attributes={'ttl': 14, 'backup_task': True})

    def on_execute(self):
        # скачать contentline learner
        contentline_learner = download_contentline_learner(self)

        # скачать greetings
        greetings_path = ArcadiaTestData.get_arcadia_test_data(
            self, 'arcadia:/arc/trunk/arcadia_tests_data/yweb/iex/packages/settings/contentline.greetings.txt')

        if self.ctx['training_set_to_model_and_test_set_to_sbs']:
            # скачать обучающую выборку
            training_set_path = self.abs_path('training_set')
            download_archive_and_extract(self.ctx['training_set_url'], training_set_path)

            # запустить contentline learner на обучающей выборке
            print_folder_tree(self, self.abs_path(), True)
            model_path = self.abs_path('contentline.model')
            run_process([contentline_learner, '--learn', '--folder', training_set_path, '--greetings', greetings_path,
                         '--model', model_path], log_prefix='learn')

            # создать ресурс с моделью
            self.create_resource(description=self.descr, resource_path=model_path,
                                 resource_type='IEX_CONTENTLINE_MODEL',
                                 attributes={'ttl': 'inf', 'backup_task': True})

            if self.ctx['testing_set_url'] != '':
                self.generate_sbs_data(contentline_learner, greetings_path, model_path)

        if self.ctx['model_and_test_set_to_sbs']:
            model_path = download_contentline_model(self)
            self.generate_sbs_data(contentline_learner, greetings_path, model_path)


__Task__ = IexUpdateContentlineModel
