# -*- coding: utf-8 -*-

import requests
import logging
import shutil
import os
import collections

from sandbox.projects.common.decorators import retries

from sandbox import sdk2

from sandbox.common.types.task import Status
from sandbox.common.types.client import Tag

from sandbox.projects import resource_types
import sandbox.projects.images.resource_types as images_resource_type
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.base_search_quality import threadPool
from sandbox.projects.common.search.settings import ImagesSettings
from sandbox.projects.common.http_responses.request import filename_for_request
from sandbox.projects.common.http_responses.request import host_and_path
from sandbox.projects.common.http_responses.BaseLocalResponsesTask import REQUESTS_RESOURCE_KEY, REQUESTS_LIMIT_KEY, \
    REQUIRE_ALL_SUCCESS_KEY, create_parameters


class GetResponseException(Exception):
    pass


@retries(max_tries=5, delay=1, exceptions=(GetResponseException, ))
def _get_request(url, headers):
    try:
        response = requests.get(url, headers=headers, timeout=10, stream=True)
        # 404 is ok
        if response.status_code != 404:
            response.raise_for_status()
        return response
    except requests.RequestException as e:
        err = "Failed to retrieve '{}' error {}".format(url, e)
        logging.info(err)
        raise GetResponseException(err)


class ImproxyGetProdResponses(SandboxTask):
    """
        Собирает выдачу продакшн-тумбнейлов
    """
    type = 'IMPROXY_GET_PROD_RESPONSES'

    client_tags = Tag.GENERIC

    input_parameters = create_parameters(images_resource_type.IMAGES_THUMBS_REQUESTS, 10000)

    output_resource_type = resource_types.THUMB_DAEMON_RESPONSES_ARCHIVE

    def get_data_dir(self):
        return self.abs_path(self.output_resource_type.data_dir)

    def get_result_dir(self):
        return self.abs_path('responses')

    def get_result_archive_path(self):
        return os.path.join(self.get_result_dir(), self.output_resource_type.archive_file)

    def on_enqueue(self):
        self._create_resource(
            self.descr,
            self.get_result_dir(),
            self.output_resource_type
        )

    def on_execute(self):
        requests_resource = sdk2.Resource.find(id=self.ctx[REQUESTS_RESOURCE_KEY]).first()
        if not requests_resource:
            requests_resource = channel.sandbox.get_resource(self.ctx[REQUESTS_RESOURCE_KEY])
            requests_task_status = channel.sandbox.get_task(requests_resource.task_id).new_status
        else:
            requests_task_status = requests_resource.task.status

        if requests_task_status not in tuple(Status.Group.FINISH + Status.Group.BREAK):
            self.wait_tasks(
                tasks=[requests_resource.task_id, ],
                statuses=tuple(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True
            )
        requests_resource_path = self.sync_resource(requests_resource.id)

        make_folder(self.get_data_dir(), True)
        make_folder(self.get_result_dir(), True)

        def response_getter(urls, *args, **kwargs):
            result = list()

            for url in urls:
                request_host, request_path = host_and_path(url)

                accept_header = '*/*'
                if request_host == ImagesSettings.IMPROXY_WEBP_HOST:
                    accept_header = 'image/webp,image/*,*/*;q=0.8'

                headers = {
                    'Host': request_host,
                    'Accept': accept_header
                }

                response = None
                try:
                    response = _get_request(url, headers)
                    response.raw.decode_content = True
                except GetResponseException as e:
                    if self.ctx[REQUIRE_ALL_SUCCESS_KEY]:
                        raise SandboxTaskFailureError("Bad response: {} error: {}".format(url, e))

                data_file_path = os.path.join(self.get_data_dir(), filename_for_request(request_path))
                with open(data_file_path, 'w') as data_file:
                    logging.info("Save response from {} to {} file".format(url, data_file_path))
                    if hasattr(response, 'raw') and response.raw is not None:
                        shutil.copyfileobj(response.raw, data_file)

                if os.stat(data_file_path).st_size == 0 and self.ctx[REQUIRE_ALL_SUCCESS_KEY]:
                    raise SandboxTaskFailureError('Got empty response with success status: {}'.format(url))

                header_dump = list()
                if hasattr(response, 'headers') and isinstance(response.headers, collections.Iterable):
                    header_dump = ["{}: {}".format(key, value) for key, value in response.headers.iteritems()]
                status_code = '000'
                if hasattr(response, 'status_code') and response.status_code is not None:
                    status_code = response.status_code
                result.append("-- {} {}\n{}{}".format(url, str(status_code), "\n".join(header_dump),
                                                      '\n' if len(header_dump) > 0 else ''))

            return result

        result_list = list()
        with open(requests_resource_path, 'r') as requests_file:
            # FIXME: file mapping into memory
            total_requests = list(map(str.strip, requests_file))
            if not total_requests or len(total_requests) < self.ctx[REQUESTS_LIMIT_KEY]:
                raise SandboxTaskFailureError("Requests's not enough")
            result_list = threadPool.process_data(response_getter, total_requests[:self.ctx[REQUESTS_LIMIT_KEY]],
                                                  params=None,
                                                  process_count=int(self.client_info['ncpu'] // 2),
                                                  use_processes=False)

        with open(os.path.join(self.get_result_dir(), self.output_resource_type.meta_file), 'w') as stats_file:
            if len(result_list) < 1:
                raise SandboxTaskFailureError('Empty result. Check logs.')
            for item in result_list:
                stats_file.write(item)

        run_process(['tar', '-czf', self.get_result_archive_path(), self.output_resource_type.data_dir],
                    wait=True, check=True)


__Task__ = ImproxyGetProdResponses
