# -*- coding: utf-8 -*-

import logging
import requests

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny

from sandbox.projects.DeployNannyShardmap import NannyServiceNameParameter
from sandbox.projects import SetItsValueTask
from sandbox.projects.common.utils import check_subtasks_fails

from sandbox.sandboxsdk.sandboxapi import Sandbox
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ImproxySetBaseVersion(SandboxTask):
    """
        Получаем версию базы у сервиса в няне и пушим ITS ручку
        с этим контентом
    """

    type = 'IMPROXY_SET_BASE_VERSION'

    input_parameters = [
        NannyServiceNameParameter,
        SetItsValueTask.ItsRuchkaNameParameter
    ]

    def _get_active_snapshot(self, nanny_client):
        """
            Получить последний активный стейт сервиса из nanny
        """
        current_state = nanny_client.get_service_current_state(
            self.ctx[NannyServiceNameParameter.name])
        if current_state[u'content'][u'summary'][u'value'] not in (u'ONLINE', u'UPDATING', u'PREPARING'):
            logging.debug("Service is offline: %s" % (self.ctx[NannyServiceNameParameter.name]))
            return None
        for snapshot in current_state[u'content'][u'active_snapshots']:
            if snapshot[u'state'] in (
                    u'ACTIVATING',
                    u'GENERATING',
                    u'PREPARING'):
                return current_state[u'content'][u'rollback_snapshot'][u'snapshot_id']
            elif snapshot[u'state'] in (u'ACTIVE'):
                return snapshot[u'snapshot_id']

    def _get_nanny_attrs_by_snapshot(self, nanny_client, snapshot_id):
        """
            Получить атрибуты рантайма по снепшоту из сервисов 2.0
            (этого нет в стандартной библиотеке nanny)
        """
        resources_url = nanny_client._api_url + \
            '/v2/history/services/runtime_attrs/' + snapshot_id
        try:
            r = nanny_client._session.get(resources_url)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise nanny._handle_http_error(e)
        except Exception as e:
            raise nanny.NannyApiException(
                'Failed to call Nanny API. Error: {}'.format(e))
        return r.json()

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token'),
        )

        subtasks = channel.sandbox.list_tasks(parent_id=self.id)
        tasks_to_wait = filter(lambda x: not x.is_done(), subtasks)
        if tasks_to_wait:
            check_subtasks_fails()

        if 'production_snapshot' not in self.ctx:
            self.ctx['production_snapshot'] = self._get_active_snapshot(nanny_client)

        if self.ctx['production_snapshot'] is not None:
            # get snapshot attr
            active_runtime_attrs = self._get_nanny_attrs_by_snapshot(
                nanny_client, self.ctx['production_snapshot'])
            service_shardmap_dump = active_runtime_attrs[
                u'content'][u'resources'][u'sandbox_bsc_shard']
            if 'SANDBOX_SHARDMAP' in service_shardmap_dump[u'chosen_type']:
                production_shardmap_task_id = service_shardmap_dump[
                    u'sandbox_shardmap'][u'task_id']
                shardmap_resource_type_str = service_shardmap_dump[
                    u'sandbox_shardmap'][u'resource_type']
                shardmap_resource_type = getattr(resource_types, shardmap_resource_type_str)
                production_resource_id = apihelpers.get_task_resource_id(
                    production_shardmap_task_id,
                    shardmap_resource_type)
                production_shard_state = Sandbox().get_resource_attribute(
                    production_resource_id,
                    u'shard_state')
                if not production_shard_state:
                    production_shard_state = Sandbox().get_resource_attribute(
                        production_resource_id,
                        u'shardmap_version')
                    if not production_shard_state:
                        raise SandboxTaskFailureError(
                            "Couldn't get sandbox id with production shardmap for service %s" %
                            (self.ctx[NannyServiceNameParameter.name]))
                self.ctx['production_shard_state'] = production_shard_state
            else:
                raise SandboxTaskFailureError(
                    "Selected service doesn't use shardmaps")

        if self.ctx['production_shard_state'] is not None and 'its_task' not in self.ctx:
            # run its ruchka
            subtask_ctx = {
                SetItsValueTask.ItsRuchkaNameParameter.name: self.ctx[SetItsValueTask.ItsRuchkaNameParameter.name],
                SetItsValueTask.ItsValueNameParameter.name: self.ctx['production_shard_state'],
            }
            its_task = self.create_subtask(
                task_type=SetItsValueTask.SetItsValueTask.type,
                description=u'',
                input_parameters=subtask_ctx)
            self.ctx['its_task'] = its_task.id
            self.wait_all_tasks_stop_executing([its_task.id])


__Task__ = ImproxySetBaseVersion
