import os
import httplib
import base64

from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.network import wait_port_is_free
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.search.components import create_improxy_params
from sandbox.projects.common.search.components import get_improxy
from sandbox.projects.common.search.settings import ImagesSettings, VideoSettings


def test_http_request_one_method(hostname, port, method, http_path, expected_code, expected_data=None, headers={},
                                 expected_headers={}):
    connection = httplib.HTTPConnection(hostname, port)
    connection.request(method, http_path, headers=headers)
    response = connection.getresponse()
    if expected_code != response.status:
        raise SandboxTaskFailureError("Request '%s', method %s, expected code %d, got %d" % (http_path,
                                                                                             method,
                                                                                             expected_code,
                                                                                             response.status))
    headers_dict = dict([(x[0].lower(), x[1]) for x in response.getheaders()])
    for expected_name, expected_value in expected_headers.iteritems():
        value = headers_dict.get(expected_name.lower(), None)
        if value != expected_value:
            raise SandboxTaskFailureError("Request '%s', method '%s', expected header '%s' value %s, got %s")

    data = response.read()
    if expected_data is not None:
        if data != expected_data:
            raise SandboxTaskFailureError("Request '%s', expected data\n'%s'\ngot\n'%s'\n" %
                                          (http_path, expected_data, data))

    return data, headers_dict


def test_http_request(port, http_path, expected_code, expected_data=None, headers={}, expected_headers={}, hostname=os.uname()[1], limit_size_in_kbytes=False, disable_headers_check=False):
    _, head_headers = test_http_request_one_method(hostname, port, 'HEAD', http_path, expected_code, '', headers)
    data, get_headers = test_http_request_one_method(hostname, port, 'GET', http_path, expected_code, expected_data, headers,
                                                     expected_headers)

    content_length_key = 'content-length'
    head_headers.pop(content_length_key, None)
    get_headers.pop(content_length_key, None)

    if head_headers != get_headers and not disable_headers_check:
        raise SandboxTaskFailureError("Request %s, headers for HEAD and GET are different:\n%s\n%s\n" %
                                      (http_path, head_headers, get_headers))

    if limit_size_in_kbytes:
        if data.__sizeof__() > (limit_size_in_kbytes * 1024):
            raise SandboxTaskFailureError("Response length is larger than limit %s (kb). Requested url: %s" %
                                          (limit_size_in_kbytes, http_path))
    return data, get_headers


class ImproxyTestFunc(SandboxTask):
    type = 'IMPROXY_TEST_FUNC'
    client_tags = Tag.LINUX_PRECISE & Tag.IPV4 & ~Tag.LXC

    input_parameters = create_improxy_params(ImagesSettings.IMPROXY_CONFIGS + VideoSettings.IMPROXY_CONFIGS)

    def on_execute(self):
        improxy = get_improxy()
        improxy.start()
        improxy.wait()

        test_http_request(improxy.port, '/check.gif', httplib.OK,
                          base64.b64decode("R0lGODlhAQABAIABAAAAAP///yH5BAEAAAEALAAAAAABAAEAAAICTAEAOw=="))
        test_http_request(improxy.port, '/robots.txt', httplib.OK,
                          "User-agent: *\nDisallow: /\n")
        test_http_request(improxy.port, '/', httplib.FOUND,
                          expected_headers={'Location': "http://www.yandex.ru/"})
        test_http_request(improxy.port, '/i?id=somecrazystuff-234-55435', httplib.NOT_FOUND)
        test_http_request(improxy.port, '/admin?action=stats', httplib.OK, hostname='localhost', limit_size_in_kbytes=5120, disable_headers_check=True)
        test_http_request(improxy.port, '/admin/events/call/report', httplib.OK, hostname='localhost', limit_size_in_kbytes=1024, disable_headers_check=True)

        test_http_request_one_method('localhost', improxy.port, 'GET', '/admin?action=shutdown', httplib.OK)
        shutdown_timeout = 90
        if not wait_port_is_free(port=improxy.port, timeout=shutdown_timeout):
            raise SandboxTaskFailureError("Could not shut down port %s in %d seconds" % (improxy.port, shutdown_timeout))


__Task__ = ImproxyTestFunc
