# -*- coding: utf-8 -*-

import os
import logging

from collections import OrderedDict

from sandbox import common
import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import ResourceSelector

from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common import dolbilka
from sandbox.projects.common.search import components
from sandbox.projects.common.search.settings import ImagesSettings, VideoSettings

CACHE_HIT_KEY = 'cache_hit'


class PlanResourceParameter(ResourceSelector):
    name = 'plan_resource_id'
    description = 'Plan'
    resource_type = resource_types.IMPROXY_PLAN
    required = True


class ImproxyTestPerformance(SandboxTask):
    """
        Тестирование производительности тумбнейлерной прокси.
        Прокся обстреливается один раз в выбранном режиме, после этого выводится результат.
        Проверяется полученный rps и количество различных http-ошибок (оборванные соединения, 404, 503 и т.п.)
    """

    type = 'IMPROXY_TEST_PERFORMANCE'
    dns = ctm.DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE & Tag.INTEL_E5_2660 & ~Tag.LXC

    execution_space = 10000

    @property
    def footer(self):
        if 'rps' in self.ctx:
            items_header = [
                {'key': 'description', 'title': '&nbsp;'},
                {'key': 'results', 'title': '&nbsp;'}
            ]
            items_body = {}
            metrics = OrderedDict([
                ('Total requests', self.ctx['requests']),
                ('Requests per second', self.ctx['rps']),
                ('Average request time', self.ctx['average_request_time']),
                ('Cache hit', self.ctx['cache_hit']),
            ])
            for key, value in metrics.items():
                items_body.setdefault('description', []).append(key)
                items_body.setdefault('results', []).append(value)
            errors = OrderedDict([
                ('Error requests', self.ctx['error_requests']),
                ('Not found', self.ctx['requests_not_found']),
                ('Proxy error (502)', self.ctx['bad_gateway']),
            ])
            for key, value in errors.items():
                if value > 0:
                    result = "<b style='color:red'>%s</b>" % (value)
                else:
                    result = "<b style='color:green'>%s</b>" % (value)
                items_body.setdefault('description', []).append(key)
                items_body.setdefault('results', []).append(result)
            return {'<h4>Results</h4>': {
                'header': items_header,
                'body': items_body
            }}

    def set_cgroups(self):
        if not utils.get_or_default(self.ctx, components.ImproxyUseCgroup):
            return
        cg = common.os.CGroup("test_task")
        cg.memory["low_limit_in_bytes"] = ImagesSettings.IMPROXY_CGROUP_LOW_LIMIT
        cg.memory["limit_in_bytes"] = ImagesSettings.IMPROXY_CGROUP_LIMIT
        cg.cpu["smart"] = 1
        return cg

    # Прогрев не требуется, поэтому параметр "число сессий" выставляем в 1 в initCtx
    input_parameters = \
        components.create_improxy_params(ImagesSettings.IMPROXY_CONFIGS + VideoSettings.IMPROXY_CONFIGS) +\
        (dolbilka.DolbilkaExecutorRequestsLimit,
         dolbilka.DolbilkaExecutorMode,
         dolbilka.DolbilkaMaximumSimultaneousRequests,
         dolbilka.DolbilkaDelayMultiplier,
         dolbilka.DolbilkaBounds,) +\
        (PlanResourceParameter, )

    def initCtx(self):
        self.ctx['dolbilka_executor_sessions'] = 1

    def on_execute(self):
        plan_path = self.sync_resource(self.ctx[PlanResourceParameter.name])

        improxy = components.get_improxy(cgroup=self.set_cgroups())
        d_executor = dolbilka.DolbilkaExecutor()
        try:
            hostname = os.uname()[1]
            result = d_executor.run_sessions(plan_path, improxy, host=hostname)[0]
            logging.info('Got raw result: {}'.format(result))
        except:
            improxy.save_logs_resource()
            raise

        for key in ['error_requests', 'rps', 'average_request_time', 'requests_not_found', 'bad_gateway', 'requests']:
            self.ctx[key] = int(float(result[key])) if result[key] is not None else 0
        self.ctx['has_errors'] = \
            self.ctx['error_requests'] + self.ctx['requests_not_found'] + self.ctx['bad_gateway'] > 0
        with open(improxy.access_log_path(), 'r') as access_log:
            cache_hits = len(filter(lambda x: 'cached' in x, access_log))
            self.ctx[CACHE_HIT_KEY] = float(cache_hits) / self.ctx['requests']

        improxy.save_logs_resource()


__Task__ = ImproxyTestPerformance
