# coding: utf-8

import os
import logging
import tarfile

from sandbox import sdk2

from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.build import parameters as ya_make_parameters
from sandbox.projects.common import constants as consts
from sandbox.projects.common.nanny import nanny

import sandbox.projects.IndexNow as index_now_resources


TARGET_RESOURCES = (
    index_now_resources.IndexNowRequestProcessor,
    index_now_resources.IndexNowServer,
)

TARGET_FILES = (
    index_now_resources.IndexNowConfigs
)


class BuildIndexNowPackage(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
    Binaries and configs for Index Now.
    '''

    class Requirements(sdk2.Task.Requirements):
        disk_space = 70 * 1024
        cores = 32
        ram = 2048

    class Parameters(sdk2.Task.Parameters):
        description = "Index now build task"

        ArcadiaUrl = ya_make_parameters.ArcadiaUrl()
        BuildSystem = ya_make_parameters.BuildSystem(default_value=consts.YMAKE_BUILD_SYSTEM)
        BuildType = ya_make_parameters.BuildType(default_value=consts.RELEASE_BUILD_TYPE)
        YtProxy = ya_make_parameters.YtProxy()
        YtDir = ya_make_parameters.YtDir()
        YtTokenVaultOwner = ya_make_parameters.YtTokenVaultOwner(default_value="UKROP-ROBOT")
        YtTokenVaultName = ya_make_parameters.YtTokenVaultName(default_value="sandbox-build-yt-token")
        YtPut = ya_make_parameters.YtPut()
        YtStoreCodec = ya_make_parameters.YtStoreCodec()
        YtReplaceResult = ya_make_parameters.YtReplaceResult()
        Sanitize = ya_make_parameters.Sanitize()
        ArcadiaPatch = ya_make_parameters.ArcadiaPatch()

        EnabledBuilds = {}
        for resource in TARGET_RESOURCES:
            target = os.path.basename(resource.arcadia_build_path)
            key = 'build_{}'.format(target)
            EnabledBuilds[key] = resource

        with sdk2.parameters.CheckGroup("Build binaries ") as build_targets:
            build_targets.choices = [(build, build) for build, resource in EnabledBuilds.items()]

        ConfigsCheckbox = sdk2.parameters.Bool("Configs ", default_value=True)

    def build(self, build_targets, source_directory, results_dir):
        arcadiasdk.do_build(
            build_system=self.Parameters.BuildSystem,
            build_type=self.Parameters.BuildType,
            source_root=source_directory,
            results_dir=results_dir,
            targets=build_targets,
            clear_build=False,
            sanitize=self.Parameters.Sanitize
        )

    def on_execute(self):
        logging.info("BuildIndexNowPackage.on_execute")
        results_dir = "build"
        targets = []

        for checked in self.Parameters.build_targets:
            targets.append(self.Parameters.EnabledBuilds[checked].arcadia_build_path)

        for target in targets:
            logging.info("Build targets: {}".format(target))

        with arcadiasdk.mount_arc_path(self.Parameters.ArcadiaUrl) as source_directory:
            if getattr(self.Parameters, 'ArcadiaPatch', ''):
                logging.info("Apply patch: {}".format(self.Parameters.ArcadiaPatch))
                arcadiasdk.apply_patch(self, source_directory, self.Parameters.ArcadiaPatch, self.path())

            if targets:
                self.build(build_targets=targets,
                           source_directory=source_directory,
                           results_dir=results_dir)

                for checked in self.Parameters.build_targets:
                    logging.info("Prepare: {}".format(checked))
                    resource = self.Parameters.EnabledBuilds[checked]
                    resource(self,
                             str(resource),
                             os.path.join(results_dir, "bin", os.path.basename(resource.arcadia_build_path)))

            if self.Parameters.ConfigsCheckbox:
                file_configs = index_now_resources.IndexNowConfigs.resource_file
                logging.info("Creating configs resource: {}".format(file_configs))
                with tarfile.open(file_configs, "w:gz") as tar:
                    tar.add(os.path.join(source_directory, "robot/samovar/conf"), arcname="conf")
                index_now_resources.IndexNowConfigs(self, "IndexNow configs", file_configs)

        logging.info("Build finished.")

    def on_finish(self, prev_status, status):
        logging.info("BuildIndexNowPackage.on_finish")
