import logging
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.advq.common.parameters import YtParameters

import Jobs.SectionsByDate
import Jobs.TemplatesByDate
import Jobs.PublishedTemplatesByDate
import Jobs.PublishedTemplatesLatest
import Jobs.PagesWithSection
import Jobs.Sites
import Jobs.Pages

logger = logging.getLogger(__name__)


class LcGetStatTask(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (
            PipEnvironment("yandex-yt"),
            PipEnvironment("psycopg2-binary"),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        lpc_psycopg2_connection_string_vault_user = sdk2.parameters.String(
            "psycopg2 connection string vault user",
            default="LPC",
        )
        lpc_psycopg2_connection_string_vault_name = sdk2.parameters.String(
            "psycopg2 connection string vault name",
            default="lpc-psycopg2-production",
        )

    def get_lpc_psycopg2_connection(self):
        if (
                self.Parameters.lpc_psycopg2_connection_string_vault_user
                and self.Parameters.lpc_psycopg2_connection_string_vault_name
        ):
            import psycopg2

            connection_string = str(sdk2.Vault.data(
                self.Parameters.lpc_psycopg2_connection_string_vault_user,
                self.Parameters.lpc_psycopg2_connection_string_vault_name,
            ))
            return psycopg2.connect(connection_string)
        else:
            return None

    def get_yt_token(self):
        return sdk2.Vault.data(
            self.Parameters.yt_token_vault_user,
            self.Parameters.yt_token_vault_name,
        )

    def on_execute(self):
        from yt import wrapper as yt

        connection = self.get_lpc_psycopg2_connection()
        yt_client = yt.YtClient(proxy=str(self.Parameters.yt_proxy), token=self.get_yt_token())

        if (connection is None):
            logger.info('Failed to connect. Quitting.')
            return

        Jobs.SectionsByDate.Job(connection, yt_client).on_execute()
        Jobs.TemplatesByDate.Job(connection, yt_client).on_execute()
        Jobs.PublishedTemplatesByDate.Job(connection, yt_client).on_execute()
        Jobs.PublishedTemplatesLatest.Job(connection, yt_client).on_execute()
        Jobs.PagesWithSection.Job(connection, yt_client).on_execute()
        Jobs.Sites.Job(connection, yt_client).on_execute()
        Jobs.Pages.Job(connection, yt_client).on_execute()
