import yt.wrapper as yt
from psycopg2 import connect
from psycopg2.extras import Json, execute_batch
from .Dumper import Dumper

PG_TABLE = "watson_vk_dump"
YT_SERVER = "hahn.yt.yandex.net"
YT_TABLE = "//home/lpc/watson_vk_dump"


def row_filter(row):
    if row["Error"] is not None or row["Result"] is None:
        return False

    products = row["Result"][0].get("products", [])

    return len(products) > 0


class VkPgDumper(Dumper):
    def __init__(self, yt_token, yt_server, yt_tables, pg_connection_string, batch_size=1000, rows_range=None):
        super(VkPgDumper, self).__init__(
            yt_token, yt_server, yt_tables, batch_size, rows_range)
        self.pg_connection_string = pg_connection_string

        self.ensure_table()

    def ensure_table(self):
        with connect(self.pg_connection_string) as conn:
            with conn.cursor() as cur:
                cur.execute("""
                    CREATE TABLE IF NOT EXISTS {}
                    (
                        id serial PRIMARY KEY,
                        url text NOT NULL UNIQUE,
                        data jsonb NOT NULL
                    );
                """.format(PG_TABLE))

    def filter(self, row):
        return row_filter(row)

    def dump(self, rows):
        query = """
            INSERT INTO {} (url, data) VALUES (%(url)s, %(data)s)
            ON CONFLICT (url) DO
            UPDATE SET data = %(data)s;
        """.format(PG_TABLE)

        with connect(self.pg_connection_string) as conn:
            with conn.cursor() as cur:
                execute_batch(cur, query, [{
                    "url": row["Url"],
                    "data": Json(row["Result"][0])
                } for row in rows])


class VkYtDumper(Dumper):
    def __init__(self, yt_token, yt_server, yt_tables, pg_connection_string, batch_size=1000, rows_range=None):
        super(VkYtDumper, self).__init__(
            yt_token, yt_server, yt_tables, batch_size, rows_range)

        self.client = yt.YtClient(proxy=YT_SERVER, token=self.yt_token)

        self.ensure_table()

    def ensure_table(self):
        if self.client.exists(YT_TABLE):
            return

        schema = [
            {"name": "url", "type": "string",
                "sort_order": "ascending", "required": True},
            {"name": "data", "type": "any"},
        ]

        self.client.create_table(YT_TABLE, attributes={
            "schema": schema, "dynamic": True})

        self.client.mount_table(YT_TABLE)

    def filter(self, row):
        return row_filter(row)

    def dump(self, rows):
        self.client.insert_rows("//home/lpc/watson_vk_dump", [{
            "url": row["Url"],
            "data": row["Result"][0]
        } for row in rows], update=True)
