import logging
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment

logger = logging.getLogger(__name__)


class LcWatsonDumperTask(sdk2.Task):
    """ Dumps Watson parser results """

    class Requirements(sdk2.Requirements):
        cores = 1
        environments = (
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet"),
            PipEnvironment("psycopg2-binary"),
        )

    class Parameters(sdk2.Task.Parameters):
        yt_token_vault_user = sdk2.parameters.String(
            "YT token vault owner",
            default="LC"
        )

        yt_token_vault_name = sdk2.parameters.String(
            "YT token vault name",
            default="yt-token"
        )

        pg_connection_string_vault_user = sdk2.parameters.String(
            "PG connection string vault user",
            default="LC"
        )

        pg_connection_string_vault_name = sdk2.parameters.String(
            "PG connection string vault name",
            default="psycopg2-testing"
        )

        yt_server = sdk2.parameters.String(
            "YT server name",
            required=True,
            default="arnold.yt.yandex.net"
        )

        yt_tables = sdk2.parameters.List(
            "YT tables",
            required=True,
            default=[
                # vk markets table
                "//home/watson/parsepl/parsers_data/nodge_vk_com_20191010_1526_40684d/kwyt_full/latest/results",
                # vk groups table
                "//home/watson/parsepl/parsers_data/nodge_vk_com_20191010_0914_234551/kwyt_full/latest/results"
            ],
        )

        with sdk2.parameters.RadioGroup("Dumper class", required=True) as dumper_class:
            dumper_class.values.VkYtDumper = "VkYtDumper"
            dumper_class.values.VkPgDumper = "VkPgDumper"

        batch_size = sdk2.parameters.Integer(
            "Batch size",
            required=True,
            default=1000
        )

        rows_range = sdk2.parameters.Bool("Export rows range")

        with rows_range.value[True]:
            rows_start = sdk2.parameters.Integer(
                "Rows range start",
                required=True,
                default=0
            )

            rows_end = sdk2.parameters.Integer(
                "Rows range end",
                required=True,
                default=1000
            )

    def on_execute(self):
        from .VkDumper import VkPgDumper, VkYtDumper

        yt_token = sdk2.Vault.data(
            self.Parameters.yt_token_vault_user,
            self.Parameters.yt_token_vault_name,
        )

        pg_connection_string = sdk2.Vault.data(
            self.Parameters.pg_connection_string_vault_user,
            self.Parameters.pg_connection_string_vault_name,
        )

        if (yt_token is None):
            logger.info("Can't get YT token from valut. Quitting")
            return

        if (pg_connection_string is None):
            logger.info("Can't get PG connection string from valut. Quitting")
            return

        yt_server = self.Parameters.yt_server
        yt_tables = self.Parameters.yt_tables
        batch_size = self.Parameters.batch_size
        dumper_class = self.Parameters.dumper_class

        rows_range = None

        if self.Parameters.rows_range:
            rows_range = (self.Parameters.rows_start, self.Parameters.rows_end)

        dumpers = {
            "VkPgDumper": VkPgDumper,
            "VkYtDumper": VkYtDumper,
        }

        dumpers[dumper_class](
            yt_token,
            yt_server,
            yt_tables,
            pg_connection_string,
            batch_size=batch_size,
            rows_range=rows_range
        ).run()
