import logging

logger = logging.getLogger(__name__)


class SchemaColumn(object):
    """
    Represents column in yt database
    :param schema - schema column definition for yt
    :param transformer - function to transform data from batch and creative to the one or more yt rows. As a default
    takes batch.get(yt_column_name)
    """

    def __init__(self, schema, transformer=None):
        self.schema = schema
        self.transformer = transformer or (lambda batch, creative: batch.get(schema["name"]))

    def transform(self, batch, creative):
        return self.transformer(batch, creative)

    def name(self):
        return self.schema["name"]


class YtDumpDriver(object):

    def __init__(self, connection, yt_client, task=None):
        self.connection = connection
        self.yt_client = yt_client
        self.task = task

    def prepareForTable(self, schema, rows):
        return [{column.name(): column.transform(row, None) for column in schema} for row in rows]

    def should_append_table(self):
        return False

    def get_task(self):
        return self.task

    def on_execute(self):
        from yt import wrapper as yt

        schema = self.get_schema()
        table_path = self.get_table()

        if not self.yt_client.exists(table_path):
            logger.info('Table %s not exists. Creating...', table_path)
            self.yt_client.create(
                "table",
                table_path,
                attributes={"schema": map(lambda column: column.schema, schema)})

        logger.info('Dumping data...')

        fetched_data = self.fetch_data(self.connection)
        logger.info('Got data to dump...')
        prepared_data = self.prepareForTable(schema, fetched_data)
        should_append = self.should_append_table()

        self.yt_client.write_table(
            self.yt_client.TablePath(table_path, append=should_append),
            prepared_data,
            format=yt.JsonFormat(attributes={"encode_utf8": False}, encoding='utf-8'),
            force_create=False,
        )
        logger.info('Dump complete...')
