import logging
from logging import error
import sandbox.sandboxsdk.task as sdk_task
import datetime
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter, ListRepeater
from sandbox.sandboxsdk.channel import channel


ML_ENGINE_DELAYS = "yabs-ml-engine-delays"


class TaskId(SandboxStringParameter):
    name = 'task_id'
    description = 'Task id'
    required = True


class AlertDelta(SandboxIntegerParameter):
    name = 'delta_days'
    description = "Max days before alert"
    default_value = 2
    required = True


class NotifyListParam(ListRepeater, SandboxStringParameter):
    name = 'notify_list_param'
    description = 'List of recipients to notify'
    required = True


FILENAME = 'conf.json'


def get_latest_resource(*args, **kwargs):
    resources = channel.sandbox.list_resources(order_by="-id", limit=1, status="READY", *args, **kwargs)
    if resources:
        return resources[0]
    error("Can't find latest resource: %s", kwargs)


class Task(sdk_task.SandboxTask):
    """ Monitor ml-engine dump, when datetime.now - AlertDelta > last_log_date sends email"""

    type = "ML_ENGINE_DUMP_MONITOR"

    input_parameters = [TaskId, AlertDelta, NotifyListParam]

    def on_execute(self):
        task_id = self.ctx[TaskId.name]
        logging.info("Checking task_id: {task_id}".format(task_id=task_id))
        text_dump_resource = get_latest_resource(
            resource_type="ML_ENGINE_DUMP",
            all_attrs={"task_id": self.ctx[TaskId.name]}
        )
        recipients = self.ctx[NotifyListParam.name]
        recipients += [ML_ENGINE_DELAYS]

        if text_dump_resource:
            last_log_date = text_dump_resource.attributes["last_log_date"]
            logging.info("Get resource with last_log_date {last_log_date}".format(last_log_date=last_log_date))
        else:
            logging.info("Sending email to {}".format(str(recipients)))
            channel.sandbox.send_email(
                recipients,
                None,
                "No dumps for {task_id}".format(task_id=task_id),
                "No dumps for {task_id}".format(task_id=task_id)
            )
            return

        delta = datetime.datetime.now() - datetime.datetime.strptime(last_log_date, "%Y%m%d")

        if delta.days > self.ctx[AlertDelta.name]:
            logging.info("Sending email to {}".format(str(recipients)))
            channel.sandbox.send_email(
                recipients, None, "Last log date for {task_id} is {last_log_date}".format(
                    task_id=task_id,
                    last_log_date=last_log_date
                ),
                "Last log date for {task_id} is {last_log_date}".format(
                    task_id=task_id,
                    last_log_date=last_log_date
                )
            )
