# -*- coding: utf-8 -*-
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.paths import make_folder, remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import resource_types
from sandbox.common import rest
import sandbox.common.types.client as ctc
from sandbox import sandboxsdk
import logging
import sandbox.projects.common.search.components.market_report as mp
import os
import errno
import time
import shutil

MARKET_SVN_PATH = "market"
BACKCTLD_SVN_PATH = "market/backctld"
PYLIB_SVN_PATH = "market/pylibrary"
PYCONTRIB_SVN_PATH = "market/contrib"


def mkdir_p(path):
    try:
        os.makedirs(path)
    except OSError as e:
        if e.errno == errno.EEXIST and os.path.isdir(path):
            pass
        else:
            raise


def symlink(existing_path, link_path):
    run_process(['ln', '-s', '-f', existing_path, link_path])


class LoadMarketReportIndex(task.SandboxTask):

    type = 'LOAD_MARKET_REPORT_INDEX'
    execution_space = 61440  # 60Gb
    input_parameters = mp.REPORT_INDEX_PARAMS
    client_tags = ctc.Tag.Group.LINUX

    @staticmethod
    def untar(tar_path, dst_path):
        run_process(['tar', '-xzf', tar_path, '-C', dst_path], log_prefix='tar')

    def sync_last_res(self, res_type, attrs={}):
        cl = rest.Client()
        attrs = attrs.copy()
        attrs['released'] = 'stable'
        res_data = cl.resource.read(type=str(res_type), attrs=attrs, state='READY', limit=1, order='-id')
        try:
            res_id = res_data['items'][0]['id']
        except IndexError:
            raise SandboxTaskFailureError("No resource %s with attrs %s found" % (res_type, attrs))
        return self.sync_resource(res_id)

    def get_uc_binary(self, dst_path):
        uc_tar_path = self.sync_last_res(resource_types.SCRIPT_BUILD_RESULT, {'tool_name': 'uc'})
        self.untar(uc_tar_path, dst_path)

    def get_reductor_binary(self, dst_path):
        reductor_tar_path = self.sync_last_res("MARKET_BACKCTLD")
        self.untar(reductor_tar_path, dst_path)

    def download_index_parts(self):
        index_dir = make_folder('index_download')
        index_gen = self.ctx[mp.IndexGeneration.name]
        index_gen_dir = os.path.join(index_dir, index_gen)

        kws = {
            'fallback_to_bb': True,
        }

        logging.info("Fetching remote data via skynet")
        sky_res = self.ctx[mp.SearchIndexLink.name].replace(' ', '')
        for sky_link in sky_res.split(','):
            r = sandboxsdk.copy.RemoteCopy(sky_link,
                                           index_gen_dir,
                                           log_dir=self.log_path())
            r(**kws)

        return index_dir

    def prepare_backctld(self, backctld_dir):
        root_dir = os.path.join(backctld_dir, 'root')
        self_dir = os.path.dirname(os.path.abspath(__file__))

        self.get_reductor_binary(backctld_dir)

        plugins_dst_dir = os.path.join(root_dir, 'etc', 'backctld', 'plugins')
        mkdir_p(plugins_dst_dir)
        shutil.copy(os.path.join(self_dir, 'marketsearch3.conf'), os.path.join(plugins_dst_dir, 'marketsearch3.conf'))

        mkdir_p(os.path.join(root_dir, 'var/lib/search'))
        symlink(os.path.join(backctld_dir, 'bin'), os.path.join(root_dir, 'bin'))

    def start_reload(self, backctld_prefix, backctld_dir, index_gen):
        back_pr = run_process(backctld_prefix + [
                              'marketsearch3',
                              'unpack_reload', index_gen,
                              '1800'],
                              work_dir=backctld_dir,
                              outs_to_pipe=True,
                              shell=True,
                              check=False)

        if back_pr.returncode != 0:
            result_st = back_pr.communicate()[1].decode('utf-8').strip().lower()
            self.create_backctld_log(backctld_dir)
            raise SandboxTaskFailureError('backctld failed to start, err: %s\
                                           see details in backctld_log' % result_st)

    def wait_reload_done(self, backctld_prefix, backctld_dir):
        unpack_ok = False
        for i in xrange(0, 10):
            p = run_process(backctld_prefix + ['marketsearch3', 'get_status'],
                            work_dir=backctld_dir,
                            outs_to_pipe=True,
                            shell=True,
                            check=False)
            pr_result = p.communicate()
            result_st = pr_result[0].decode('utf-8').strip().lower()
            err_result = pr_result[1].decode('utf-8')
            if p.returncode != 0:
                raise SandboxTaskFailureError('falied to check backctld status, err: %s' % err_result)

            if result_st == 'ok':
                unpack_ok = True
                break
            elif 'failed' in result_st:
                raise SandboxTaskFailureError(
                    "Backctld task failed. Info: '%s'. See more details in backctld_log" % result_st)
            else:
                logging.info("Waiting for backctld to finish. Current status:'%s', err: '%s'. Attempt: %s", result_st, err_result, i)
                time.sleep(60)

        self.create_backctld_log(backctld_dir)

        if not unpack_ok:
            raise SandboxTaskFailureError("Exceeded max wait attempts. See more details in backctld_log")

    @staticmethod
    def fix_symlinks(path):
        # move relative symlinks to absolute ones
        p = run_process(['find', path, '-type', 'l'],
                        outs_to_pipe=True,
                        shell=True,
                        check=True)
        link_result = p.communicate()
        link_result_st = link_result[0].decode('utf-8')
        for link_path in link_result_st.splitlines():
            real_link = os.path.realpath(link_path)
            os.unlink(link_path)
            os.symlink(real_link, link_path)

    @staticmethod
    def move_snippet_back_to_index(path):
        for shard_path in ['index', 'index/model']:
            snippet_shard = os.path.join(path,
                                         shard_path,
                                         'snippet-part-0')
            shard = os.path.join(path, shard_path, 'part-0')
            if os.path.exists(snippet_shard) and os.path.exists(shard):
                run_process(['mv %s/*  %s' % (snippet_shard, shard)],
                            shell=True, check=True)
                remove_path(snippet_shard)

    def unpack_index(self):
        backctld_dir = make_folder('backctld')
        backctld_bin = os.path.join(backctld_dir, "bin", "backctld")
        backctld_prefix = [backctld_bin, "--prefix", os.path.join(backctld_dir, "root")]

        self.prepare_backctld(backctld_dir)

        index_output_dir = make_folder('output_index')
        symlink(index_output_dir, os.path.join(backctld_dir, 'root/var/lib/sandbox_search'))

        download_dir = self.download_index_parts()
        symlink(download_dir, os.path.join(backctld_dir, 'root/var/lib/download'))

        self.prepare_backctld(backctld_dir)

        index_gen = self.ctx[mp.IndexGeneration.name]
        self.start_reload(backctld_prefix, backctld_dir, index_gen)
        self.wait_reload_done(backctld_prefix, backctld_dir)

        self.fix_symlinks(index_output_dir)
        self.move_snippet_back_to_index(index_output_dir)

        # Make index resource
        attrs = {}
        if self.ctx[mp.MarkWithTag.name]:
            attrs[self.ctx[mp.MarkWithTag.name]] = '.'

        self.create_resource(
            index_gen,
            'output_index',
            mp.Index.resource_type,
            attributes=attrs)

    def create_backctld_log(self, tmp_log_path):
        backctld_log_dir = make_folder('backctld_logs')
        run_process(['cp -R %s  %s' % (os.path.join(tmp_log_path, 'root/var/log/backctld'),
                     backctld_log_dir)],
                    shell=True, check=False)
        self.create_resource('backctld_log',
                             backctld_log_dir,
                             resource_types.OTHER_RESOURCE)

    def on_execute(self):
        self.unpack_index()


__Task__ = LoadMarketReportIndex
