#!/usr/bin/env python2
#  -*- coding: utf-8 -*-

from __future__ import division
from __future__ import unicode_literals
import requests
import re
import logging
from datetime import datetime
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
import tarfile
import os

from sandbox.projects.LoadStartrekSPIDashboardsToStat.resource_types import StartrekSPIDashboardsStats


class Widget(object):
    def __init__(self, token, conf, data=None):
        self._token = token
        self._conf = conf
        if isinstance(data, WidgetData):
            self._data = data
        else:
            self._data = WidgetData(self._token, self.id)

    @property
    def id(self):
        return self._conf['id']

    @property
    def query(self):
        return self._conf['filterParams']['query']

    @property
    def data(self):
        return self._data


class Widgets(object):

    def __init__(self, token):
        self._token = token
        self._widgets = {}

    def __getitem__(self, key):
        return self._widgets[key]

    def get(self, key, default=None):
        result = self._widgets.get(key)
        if result is not None:
            return result
        elif isinstance(default, Widget):
            return default
        else:
            return Widget(self._token, {'id': 'dummy'}, data=WidgetData(self._token, 'dummy', data={}))

    def __setitem__(self, key, value):
        self._widgets[key] = value

    def add(self, value):
        self._widgets[value.id] = value


class WidgetData(object):

    def __init__(self, token, widget_id, data=None):
        self._token = token
        self._id = widget_id
        self._data = {}
        if data is None:
            self._load()
        else:
            self._data = data

    def _load(self):
        headers = {
            'Authorization': 'OAuth {auth_token}'.format(auth_token=self._token)
        }
        url = 'https://st-api.yandex-team.ru/front/widgets/{widget_id}/data'.format(widget_id=self._id)
        widget_data = requests.get(url, headers=headers)
        widget_data.raise_for_status()
        response_json = widget_data.json()
        if 'data' in response_json and 'counts' in response_json['data']:
            for entry in response_json['data']['counts']:
                cleaned_from = entry["from"][:-6]
                cleaned_to = entry["to"][:-6]
                key = (
                    datetime.strptime(cleaned_from, '%Y-%m-%dT%H:%M:%S'),
                    datetime.strptime(cleaned_to, '%Y-%m-%dT%H:%M:%S')
                )
                self._data[key] = entry['count']

    def __getitem__(self, key):
        return self._data[key]

    def get(self, key, default=None):
        return self._data.get(key, default)

    def __add__(self, other):
        all_keys = list(self._data.keys())
        all_keys.extend(other._data.keys())
        all_keys = list(set(all_keys))
        res_data = {}
        for key in all_keys:
            res_data[key] = self.get(key, 0) + other.get(key, 0)
        return WidgetData(self._token, self._id, res_data)

    def __mul__(self, other):
        if not isinstance(other, int):
            raise NotImplementedError('We may only multiply with int')
        res_data = {}
        for key, value in self._data.items():
            res_data[key] = value * other
        return WidgetData(self._token, self._id, res_data)

    def __truediv__(self, other):
        all_keys = list(self._data.keys())
        all_keys.extend(other._data.keys())
        all_keys = list(set(all_keys))
        res_data = {}
        for key in all_keys:
            if other.get(key, 0) == 0:
                res_data[key] = 0
            else:
                res_data[key] = self.get(key, 0) / other.get(key, 0)
        return WidgetData(self._token, self._id, res_data)

    def round(self, ndigits=2):
        if not isinstance(ndigits, int):
            raise ValueError('ndigits must be int')
        res_data = {}
        for key, value in self._data.items():
            res_data[key] = round(value, ndigits)
        return WidgetData(self._token, self._id, res_data)

    def items(self):
        return self._data.items()


class Dashboard(object):

    def __init__(self, token, id_or_url):
        logging.debug('Initializing dashboard with {dash_id}'.format(dash_id=id_or_url))
        if id_or_url.startswith('http'):
            id_or_url = id_or_url.split('/')[-1]

        self._id = id_or_url
        self._token = token
        self._conf = self._load_conf()
        self._widgets = Widgets(self._token)

    def _load_conf(self):
        headers = {
            'Authorization': 'OAuth {auth_token}'.format(auth_token=self._token)
        }
        params = {
            'embed': 'widgets',
            'expand': 'widgets'
        }
        url = 'https://st-api.yandex-team.ru/front/dashboards/{dasboard_id}'.format(dasboard_id=self._id)
        dash_conf = requests.get(url, params=params, headers=headers)
        dash_conf.raise_for_status()
        return dash_conf.json()

    @property
    def name(self):
        return self._conf['data']['name']

    def _add_widget_by_description(self, descr):
        result = []
        pattern = re.compile(descr, flags=re.IGNORECASE)
        for widget in self._conf['data']['widgets']:
            res = pattern.search(widget['description'])
            if res:
                result.append(widget)
        if len(result) == 1:
            self._widgets[descr] = Widget(self._token, result[0])
        elif len(result) == 0:
            logging.warning(
                u'Widget not found in dashboard %s by description %s', self.name, descr)
        else:
            logging.warning(
                u'Found more than 1 widget in dashboard %s by desscription %s', self.name, descr)
        return {}

    def add_widgets(self, descriptions):
        for descr in descriptions:
            self._add_widget_by_description(descr)

    @property
    def widgets(self):
        return self._widgets

    @property
    def report(self):
        result = {
            'resolved_percent': (
                self.widgets.get(u'Количество разобранных инцидентов').data
                / self.widgets.get(u'Открытые и закрытые тикеты - когда их заводили').data) * 100,
            'problems_assigned': (
                self.widgets.get(u'Опроблеман[^ ]+ инцидент').data
                / self.widgets.get(u'Открытые и закрытые тикеты - когда их заводили').data) * 100,
            'ydt': (
                self.widgets.get(u'Учтенные в YDT').data
                / self.widgets.get(u'Открытые и закрытые тикеты - когда их заводили').data) * 100,
            'engagement': (
                self.widgets.get(u'Инциденты, заведенные командой').data
                / (
                    self.widgets.get(u'Инциденты, заведенные командой').data
                    + self.widgets.get(u'Инциденты, которые принесли Марти').data)) * 100,
            'impact': (
                self.widgets.get(u'Impact:internal').data
                / (self.widgets.get(u'Impact:internal').data + self.widgets.get(u'Impact:external').data)) * 100
        }
        result2 = Report(name=self.name)
        result2.append_data('resolved', result['resolved_percent'].round(2))
        result2.append_data('problems', result['problems_assigned'].round(2))
        result2.append_data('ydted', result['ydt'].round(2))
        result2.append_data('engagement', result['engagement'].round(2))
        result2.append_data('ais', self.widgets[u'Когда были созданы протегированные action items'].data.round(2))
        result2.append_data('impact', result['impact'].round(2))
        result2.append_data('absolute', self.widgets[u'Открытые и закрытые тикеты - когда их заводили'].data.round(2))
        return result2


class ReportRow(object):

    def __init__(
        self,
        month=0,
        dashboard='',
        resolved=0,
        problems=0,
        ydted=0,
        engagement=0,
        ais=0,
        impact=0,
        absolute=0,
    ):
        self.month = month
        self.dashboard = dashboard
        self.resolved = resolved
        self.problems = problems
        self.ydted = ydted
        self.engagement = engagement
        self.ais = ais
        self.impact = impact
        self.absolute = absolute

    @property
    def csv(self):
        return '{month};{dashboard};{resolved};{problems};{ydted};{engagement};{ais};{impact};{absolute}'.format(
            month=self.month,
            dashboard=self.dashboard,
            resolved=self.resolved,
            problems=self.problems,
            ydted=self.ydted,
            engagement=self.engagement,
            ais=self.ais,
            impact=self.impact,
            absolute=self.absolute,
        )

    @property
    def dict(self):
        return {
            'fielddate': self.month,
            'dashboard': self.dashboard,
            'resolved': self.resolved,
            'problems': self.problems,
            'ydted': self.ydted,
            'engagement': self.engagement,
            'ais': self.ais,
            'impact': self.impact,
            'absolute': self.absolute,
        }


class Report(object):

    def __init__(self, name=None):
        super(Report, self).__init__()
        self._name = name or ''
        self._report = {}

    def append_data(self, field, data):
        for dates, value in data.items():
            month = dates[0].strftime("%Y-%m-%d")
            if month not in self._report:
                self._report[month] = {'month': month}
            self._report[month][field] = value

    @property
    def rows(self):
        result = []
        for row in sorted(self._report.values(), key=lambda x: x['month']):
            if 'dashboard' not in row:
                row['dashboard'] = self._name
            result.append(ReportRow(**row))
        return result

    @property
    def csv(self):
        result = ['month;resolved;problems;ydted;engagement;ais;impact;absolute']
        result.extend(map(lambda x: x.csv, self.rows))
        return '\n'.join(result)

    @property
    def list(self):
        # was fixed by mvel@, maybe buggy
        result = []
        for row in sorted(self._report.values(), key=lambda x: x['month']):
            result.append()
        return result

    def to_stat(self, stat_token, upload_path):
        _headers = {'Authorization': 'OAuth {token}'.format(token=stat_token)}
        data = []
        for row in self.rows:
            data.append(row.dict)

        logging.info('Try to send data to statface to {path}'.format(path=upload_path))
        resp = requests.post(
            '{path}?scale=monthly'.format(path=upload_path),
            headers=_headers,
            json={
                'data': data,
            }
        )
        logging.info('Data sent to statface, response code is {status_code}'.format(status_code=resp.status_code))
        if resp.status_code != 200:
            logging.info('{text}'.format(text=resp.text))
            if resp.status_code == 400:
                logging.error('Bad request')
                logging.info(data)
            return False
        resp.raise_for_status()
        return True


class SpiDashStatsCollectorParameters(sdk2.Parameters):
    with sdk2.parameters.Group('SPI Dash Collector parameters') as dash_bundle:
        dashboards_ids = sdk2.parameters.List('Dashboards IDs')
        statface_api_addr = sdk2.parameters.String('Statface API naddress')
        report_name = sdk2.parameters.String('Report name')
        load_to_stat = sdk2.parameters.Bool("Load report to statface")
        st_token_owner = sdk2.parameters.String("Startrek token owner")
        st_token_name = sdk2.parameters.String("Startrek token name")
        stat_token_owner = sdk2.parameters.String("Stat token owner")
        stat_token_name = sdk2.parameters.String("Stat token name")


class SpiDashStatsCollector(sdk2.Task):
    """
        Collect statistics from Startrek dashboard
    """
    name_for_humans = "Spi dashboards stats collector"
    resource_filename = 'dashboards_reports.tar'

    Parameters = SpiDashStatsCollectorParameters

    class Requirements(sdk2.Requirements):
        disk_space = 1000  # 1G
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client

    class Context(sdk2.Task.Context):
        loaded_dashboards = []

    def on_execute(self):
        logging.info('Obtaining ST token')
        self.st_token = sdk2.Vault.data(
            self.Parameters.st_token_owner,
            self.Parameters.st_token_name
        )

        _path_ = self.Context.resource_path

        if not _path_:
            _path_ = os.getcwd()

        os.chdir(_path_)

        reports = []
        files = []
        for dashboard_id in self.Parameters.dashboards_ids:
            dashboard = Dashboard(self.st_token, dashboard_id)
            filename = dashboard.name
            self.Context.loaded_dashboards.append(dashboard.name)
            dashboard.add_widgets((
                u'Открытые и закрытые тикеты - когда их заводили',
                u'Количество разобранных инцидентов',
                u'(?u)Опроблеман[^ ]+ инцидент',
                u'Учтенные в YDT',
                u'Инциденты, заведенные командой',
                u'Инциденты, которые принесли Марти',
                u'Когда были созданы протегированные action items',
                u'Impact:internal',
                u'Impact:external',
            ))
            result2 = dashboard.report
            reports.append(result2)
            files.append(filename)

            with open(filename, 'w') as _f:
                _f.write(result2.csv)
        with tarfile.TarFile(self.resource_filename, 'w') as _tar_:
            for filename in files:
                logging.info('Appending {filename}'.format(filename=filename))
                _tar_.add(filename)

        StartrekSPIDashboardsStats(self, self.name_for_humans, _path_ + '/' + self.resource_filename)

        if self.Parameters.load_to_stat:
            logging.info('Obtaining Stat token')
            self.stat_token = sdk2.Vault.data(
                self.Parameters.stat_token_owner,
                self.Parameters.stat_token_name
            )

            self.upload_path = 'https://upload.{api_addr}/_api/report/data/{report_name}'.format(
                api_addr=self.Parameters.statface_api_addr,
                report_name=self.Parameters.report_name
            )
            for report in reports:
                report.to_stat(self.stat_token, self.upload_path)

        return True
