# -*- coding: utf-8 -*-

import os
import re

from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task

from sandbox.projects import resource_types


class GeoDBOps(parameters.ResourceSelector):
    name = 'geodb_builder_resource_id'
    required = True
    description = 'Tool to build GeoDB, see https://nda.ya.ru/3QzKVp'
    resource_type = resource_types.GEODB_BUILDER_EXECUTABLE


class GeoExportUrl(parameters.SandboxUrlParameter):
    name = 'geoexport_url'
    required = True
    description = 'HTTP geoexport host, see https://nda.ya.ru/3QzKVt'
    default_value = 'geoexport.yandex.ru'


class WithoutNames(parameters.SandboxBoolParameter):
    name = 'without_names'
    description = 'Omit region names'


class WithoutLocation(parameters.SandboxBoolParameter):
    name = 'without_location'
    description = 'Omit locations'


class WithoutSpan(parameters.SandboxBoolParameter):
    name = 'without_span'
    description = 'Omit span'


class WithoutPhoneCodes(parameters.SandboxBoolParameter):
    name = 'without_phone_codes'
    description = 'Omit phone codes'


class WithoutTimeZone(parameters.SandboxBoolParameter):
    name = 'without_time_zone'
    description = 'Omit time zones'


class WithoutShorName(parameters.SandboxBoolParameter):
    name = 'without_short_name'
    description = 'Omit short name'


class WithoutPopulation(parameters.SandboxBoolParameter):
    name = 'without_population'
    description = 'Omit population'


class WithoutAmbiguous(parameters.SandboxBoolParameter):
    name = 'without_ambiguous'
    description = 'Omit ambiguation data'


class WithoutServices(parameters.SandboxBoolParameter):
    name = 'without_services'
    description = 'Omit services data'


class WithoutNativeNames(parameters.SandboxBoolParameter):
    name = 'without_native_names'
    description = 'Omit native names'


class WithoutSynonymNames(parameters.SandboxBoolParameter):
    name = 'without_synonym_names'
    description = 'Omit synonym names'


GEODB_OPS_PARAMETERS = [
    GeoExportUrl,
    WithoutNames,
    WithoutLocation,
    WithoutSpan,
    WithoutPhoneCodes,
    WithoutTimeZone,
    WithoutShorName,
    WithoutPopulation,
    WithoutAmbiguous,
    WithoutServices,
    WithoutNativeNames,
    WithoutSynonymNames,
]


class MakeGeoDBData2(task.SandboxTask):

    """
        Build GeoDB data from geobase JSON export.
    """

    type = 'MAKE_GEODB_DATA_2'
    client_tags = Tag.LINUX_PRECISE
    cores = 1
    execution_space = 200  # 200 Mb
    required_ram = 3 * 1024  # 3 Gb
    input_parameters = [GeoDBOps] + GEODB_OPS_PARAMETERS

    def initCtx(self):
        self.ctx['kill_timeout'] = 5 * 60  # 5 min

    def _create_geodb_ops_cmd(self, geodb_ops_path, geodb_bin_path,
                              geobase_export_json_path, geoexport_url,
                              ):
        cmd = [geodb_ops_path, 'build', 'remote',
               '--output', geodb_bin_path,
               '--as-json', geobase_export_json_path,
               '--server', geoexport_url,
               '--verbose'
               ]

        if self.ctx[WithoutNames.name]:
            cmd.append('--without-names')

        if self.ctx[WithoutLocation.name]:
            cmd.append('--without-location')

        if self.ctx[WithoutSpan.name]:
            cmd.append('--without-span')

        if self.ctx[WithoutPhoneCodes.name]:
            cmd.append('--without-phone-codes')

        if self.ctx[WithoutTimeZone.name]:
            cmd.append('--without-time-zone')

        if self.ctx[WithoutShorName.name]:
            cmd.append('--without-short-name')

        if self.ctx[WithoutPopulation.name]:
            cmd.append('--without-population')

        if self.ctx[WithoutAmbiguous.name]:
            cmd.append('--without-ambiguous')

        if self.ctx[WithoutServices.name]:
            cmd.append('--without-services')

        if self.ctx[WithoutNativeNames.name]:
            cmd.append('--without-native-name')

        if self.ctx[WithoutSynonymNames.name]:
            cmd.append('--without-synonym-names')

        return cmd

    def _get_geodb_ops_revision(self, geodb_ops_path):
        cmd = (geodb_ops_path, '--svnrevision', )
        path = paths.get_unique_file_name(folder=paths.make_folder('junk'),
                                          file_name='geodb_ops.svnrevision')
        with open(path, 'w') as output_:
            process.run_process(cmd, stdout=output_)

        with open(path, 'r') as input_:
            data = input_.read()

        REVISION_RE = re.compile(r'Last Changed Rev: (\d+)')
        URL_RE = re.compile(r'URL: (.+)')

        revisions = REVISION_RE.findall(data)
        urls = URL_RE.findall(data)

        return revisions[0], urls[0]

    def _localize_cmd(self, cmd):
        res = []
        for x in cmd:
            if os.path.exists(x) and os.path.isfile(x):
                res.append(os.path.basename(x))
            else:
                res.append(x)

        return res

    def _make_geodb_description(self, cmd, geodb_ops_path, geodb_ops_rid):
        descr = dict()
        descr['cmd'] = self._localize_cmd(cmd)
        descr['builder_info'] = dict()
        descr['builder_info']['rid'] = geodb_ops_rid
        (descr['builder_info']['revision'],
         descr['builder_info']['url'], ) = self._get_geodb_ops_revision(geodb_ops_path)
        descr['has'] = dict()
        descr['has']['names'] = not self.ctx[WithoutNames.name]
        descr['has']['location'] = not self.ctx[WithoutLocation.name]
        descr['has']['span'] = not self.ctx[WithoutSpan.name]
        descr['has']['phone_codes'] = not self.ctx[WithoutPhoneCodes.name]
        descr['has']['time_zone'] = not self.ctx[WithoutTimeZone.name]
        descr['has']['short_name'] = not self.ctx[WithoutShorName.name]
        descr['has']['population'] = not self.ctx[WithoutPopulation.name]
        descr['has']['ambiguous'] = not self.ctx[WithoutAmbiguous.name]
        descr['has']['services'] = not self.ctx[WithoutServices.name]
        descr['has']['native_names'] = not self.ctx[WithoutNativeNames.name]
        descr['has']['synonym_names'] = not self.ctx[WithoutSynonymNames.name]

        return descr

    def on_execute(self):
        # load resources
        geodb_ops_rid = self.ctx[GeoDBOps.name]
        geodb_ops_path = self.sync_resource(self.ctx[GeoDBOps.name])

        # create resources
        geodb_bin = self.create_resource(
            description='GeoDB data',
            resource_path='geodb.bin',
            resource_type=resource_types.GEODB_DATA,
            attributes={'ttl': 90}
        )
        self.ctx['geodb.bin'] = geodb_bin.id

        geobase_export_json = self.create_resource(
            description='Geobase export in JSON format',
            resource_path='geobase_export.json',
            resource_type=resource_types.TASK_CUSTOM_LOGS,
            attributes={'ttl': 90}
        )
        geodb_ops_log = self.create_resource(
            description='geodb_ops execution log',
            resource_path='geodb_ops.log',
            resource_type=resource_types.TASK_CUSTOM_LOGS,
            attributes={'ttl': 90}
        )

        # actually run `geodb_ops`
        cmd = self._create_geodb_ops_cmd(
            geodb_ops_path, geodb_bin.path,
            geobase_export_json.path,
            self.ctx[GeoExportUrl.name]
        )
        with open(geodb_ops_log.path, 'w') as stderr:
            process.run_process(cmd, stderr=stderr)

        self.mark_resource_ready(geodb_bin.id)

        geodb_bin_descr = self._make_geodb_description(
            cmd=cmd,
            geodb_ops_path=geodb_ops_path,
            geodb_ops_rid=geodb_ops_rid
        )
        channel.sandbox.set_resource_attribute(
            resource_id=geodb_bin.id,
            attribute_name='geodb_descr',
            attribute_value=geodb_bin_descr
        )


__Task__ = MakeGeoDBData2
