# -*- coding: utf-8 -*-

import logging
import os

from sandbox.common.errors import TaskError
from sandbox.common.types.client import Tag
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import task

from sandbox.projects.common.build import YaMake
from sandbox.projects import MakeGeoDBData2
from sandbox.projects import resource_types

_MAKE_GEODB_OPS_TID_KEY = 'ya_make_tid'
_GEODB_OPS_RID_KEY = 'geodb_ops_rid'
_MAKE_GEODB_DATA_TID_KEY = 'make_geodb_data_tid'


class ArcadiaURL(parameters.SandboxArcadiaUrlParameter):
    pass


class MakeGeoDBDataFromRevision(task.SandboxTask):

    """
        Build GeoDB data using `geodb_ops` from particular revision
    """

    type = 'MAKE_GEODB_DATA_FROM_REVISION'

    client_tags = Tag.LINUX_PRECISE
    cores = 1
    execution_space = 200  # 200 Mb
    required_ram = 200  # 200 Mb
    input_parameters = [ArcadiaURL] + MakeGeoDBData2.GEODB_OPS_PARAMETERS

    def initCtx(self):
        self.ctx['kill_timeout'] = 5 * 60  # 5 min

    def _create_geodb_ops_resource(self, geodb_ops_path):
        logging.info('Creating resource with %s', geodb_ops_path)
        resource = self.create_resource(
            description='geodb_ops',
            resource_path=geodb_ops_path,
            resource_type=resource_types.GEODB_BUILDER_EXECUTABLE,
            attributes={'ttl': 90,
                        'arch': self.client_info['arch']}
        )
        self.mark_resource_ready(resource)
        logging.info('Done creating resource, resource ID is %d', resource.id)
        self.ctx[_GEODB_OPS_RID_KEY] = resource.id

    def _run_make_geodb_data_subtask(self, geodb_ops_rid):
        ctx = {}
        for x in MakeGeoDBData2.GEODB_OPS_PARAMETERS:
            if x.name in self.ctx:
                ctx[x.name] = self.ctx[x.name]

        ctx[MakeGeoDBData2.GeoDBOps.name] = geodb_ops_rid

        logging.info('Running subtask to create geodb.bin')
        subtask = self.create_subtask(
            task_type=MakeGeoDBData2.MakeGeoDBData2.type,
            description='Make GeoDB data',
            input_parameters=ctx
        )
        self.ctx[_MAKE_GEODB_DATA_TID_KEY] = subtask.id
        self.wait_task_completed(subtask)
        logging.info('Done running subtask')

    def _bake_geodb_ops(self):
        ctx = {ArcadiaURL.name: self.ctx[ArcadiaURL.name],
               'targets': 'kernel/geodb/ut;tools/geodb_ops/tests;tools/geodb_ops',
               'test': True,
               'arts': 'tools/geodb_ops/geodb_ops',
               'build_system': 'ya',
               'build_type': 'release',
               'target_platform': 'default-linux-x86_64',
               }
        logging.info('Building and testing geodb_ops in subtask')
        subtask = self.create_subtask(
            task_type=YaMake.YaMakeTask.type,
            description='Build and test geodb_ops',
            input_parameters=ctx
        )
        self.ctx[_MAKE_GEODB_OPS_TID_KEY] = subtask.id
        self.wait_task_completed(subtask)
        logging.info('Done building and testing geodb_ops')

        return subtask.id

    def _fetch_geodb_ops(self, ya_make_task_id):
        logging.info('Fetching geodb_ops')
        resource = channel.sandbox.list_resources(
            resource_type=resource_types.ARCADIA_PROJECT,
            task_id=ya_make_task_id
        )[0].id
        dir_path = self.sync_resource(resource)
        geodb_ops_path = os.path.join(paths.make_folder('bin'), 'geodb_ops')
        paths.copy_path(os.path.join(dir_path, 'geodb_ops'), geodb_ops_path)
        logging.info('Done fetching geodb_ops')
        return geodb_ops_path

    def _check_if_task_is_successfull(self, tid):
        task = channel.sandbox.get_task(tid)
        if task.new_status != self.Status.SUCCESS:
            raise TaskError('Subtask {} status is {}, not SUCCESS'.format(tid, task.status))

    def on_execute(self):
        with self.memoize_stage.bake_geodb_ops:
            self._bake_geodb_ops()

        with self.memoize_stage.create_geodb_ops_resource:
            self._check_if_task_is_successfull(self.ctx[_MAKE_GEODB_OPS_TID_KEY])
            geodb_ops_path = self._fetch_geodb_ops(self.ctx[_MAKE_GEODB_OPS_TID_KEY])
            self._create_geodb_ops_resource(geodb_ops_path)

        with self.memoize_stage.bake_geodb_data:
            self._run_make_geodb_data_subtask(self.ctx[_GEODB_OPS_RID_KEY])

        with self.memoize_stage.check_make_geodb_data_success:
            self._check_if_task_is_successfull(self.ctx[_MAKE_GEODB_DATA_TID_KEY])


__Task__ = MakeGeoDBDataFromRevision
