# -*- coding: utf-8 -*-

import logging
import re
import subprocess

from contextlib import closing

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import Container

from sandbox.sandboxsdk.channel import channel

from sandbox.sandboxsdk.paths import get_unique_file_name

from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import string


class PreInstallCommandsParameter(SandboxStringParameter):
    name = 'pre_install_commands'
    description = 'Pre-installation commands (kludges): '
    required = False
    group = 'Make maps packages list parameters'
    multiline = True


class StubbedPackagesParameter(SandboxStringParameter):
    name = 'stubbed_package_list'
    description = 'Install stubs for packages: '
    required = False
    group = 'Make maps packages list parameters'
    multiline = True


class ManualPackageListStableParameter(SandboxStringParameter):
    name = 'manual_package_list_stable'
    description = 'Needed package list from stable: '
    required = False
    group = 'Make maps packages list parameters'
    multiline = True


class ManualPackageListTestingParameter(SandboxStringParameter):
    name = 'manual_package_list_testing'
    description = 'Needed package list from testing: '
    required = False
    group = 'Make maps packages list parameters'
    multiline = True


class ManualPackageListUnstableParameter(SandboxStringParameter):
    name = 'manual_package_list_unstable'
    description = 'Needed package list from unstable: '
    required = False
    group = 'Make maps packages list parameters'
    multiline = True


class SetAttrsParameter(SandboxStringParameter):
    name = 'resource_attrs'
    description = 'Set attrs to resources (e.g.: attr1=v1, attr2=v2): '
    required = False
    group = 'Make maps packages list parameters'


class LXCImage(Container):
    name = 'lxc_image'
    description = 'Empty lxc image'
    required = True
    default_value = '305005465'


class Repository:
    STABLE = 'stable'
    TESTING = 'testing'
    UNSTABLE = 'unstable'


class PackageDescr:
    def __init__(self, name=None, version=None, repository=None):
        self.name = name
        self.version = version
        self.repository = repository


class PackageInfoBase:
    def __init__(self):
        pass

    def virtual(self):
        pass


class PackageInfo(PackageInfoBase, PackageDescr):
    def __init__(self, name=None, version=None, repository=None):
        PackageDescr.__init__(self, name, version, repository)
        self.dependencies = []
        self.conflicts = []

    def virtual(self):
        return False


class VirtualPackageInfo(PackageInfoBase):
    def __init__(self, name=None):
        self.name = name
        self.provided_by = []

    def virtual(self):
        return True


class PackageDependency:
    def __init__(self, name=None, op=None, version=None):
        self.name = name
        self.op = op
        self.version = version


class MakeMapsPackagesList(SandboxTask):
    """
        Make dependency list for yandex-packages
    """
    type = 'MAKE_MAPS_PACKAGES_LIST'

    required_ram = 50 << 10
    cores = 17
    client_tags = ctc.Tag.LINUX_PRECISE
    privileged = True

    input_parameters = (
        PreInstallCommandsParameter,
        StubbedPackagesParameter,
        ManualPackageListStableParameter,
        ManualPackageListTestingParameter,
        ManualPackageListUnstableParameter,
        SetAttrsParameter,
        LXCImage,
    )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)

        channel.task = self
        self.ctx['out_resource_id'] = self.create_resource(
            self.descr,
            'packages_list',
            'MAPS_SEARCH_PACKAGES_LIST',
            arch='any'
        ).id

    def exec_cmd(self, cmd, log_file=None):
        # cmd = "LANG=en_US.UTF-8 " + cmd
        cmd = "LANG=C " + cmd
        if log_file:
            log_file.write('SANDBOX: Executing: {0}\n'.format(cmd))
        p = subprocess.Popen(cmd, shell=True, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)

        stdout, stderr = p.communicate()
        if log_file:
            log_file.write('Stdout: << __STDOUT__ \n')
            log_file.write('{0}\n'.format(stdout))
            log_file.write('__STDOUT__\n')

        if p.returncode != 0:
            if log_file:
                log_file.write('Died with returncode {0}\n'.format(p.returncode))
            raise SandboxTaskFailureError("Command '{0}' died with exit code {1}".format(cmd, p.returncode))
        return stdout

    def choose_right_package_version(self, pkg_name, repository, log_file=None):
        pkg_info_list = self.get_package_info(pkg_name, log_file=log_file)
        if type(pkg_info_list) != list:
            return pkg_info_list

        # В stable могут находиться более старшие версии пакетов, чем в testing
        # Если так, то надо брать версию из stable
        if repository == Repository.STABLE:
            right_repos = set((Repository.STABLE, ))
        elif repository == Repository.TESTING:
            right_repos = set((Repository.STABLE, Repository.TESTING, ))
        elif repository == Repository.UNSTABLE:
            right_repos = set((Repository.STABLE, Repository.TESTING, Repository.UNSTABLE, ))

        for p in pkg_info_list:
            if p.repository in right_repos:
                return p

        return None

    def get_package_info(self, pkg_name, version=None, log_file=None):
        if version:
            cmd = 'aptitude show -v {0}={1}'.format(pkg_name, version)
        else:
            cmd = 'aptitude show -v {0}'.format(pkg_name)
        output_str = self.exec_cmd(cmd, log_file=log_file)

        if 'State: not a real package' in output_str:
            # Виртуальный пакет
            pkg_info = VirtualPackageInfo(pkg_name)
            m = re.search(r'^Provided by:(.*$\n(?:^\s.*$\n)*)$', output_str, flags=re.M)
            if not m:
                raise SandboxTaskFailureError('Cannot parse description for package {0}'.format(pkg_name))
            provided_by_str = m.group(1)

            r = re.compile(r'(\S+)\s+\((\S+)\)')
            for s in provided_by_str.split(','):
                m = r.match(s.strip())
                if not m:
                    raise SandboxTaskFailureError('Cannot parse description for package {0}'.format(pkg_name))
                dep = PackageDependency(m.group(1), '==', m.group(2))
                pkg_info.provided_by.append(dep)
            return pkg_info

        # Реальный пакет
        pkg_info_list = []
        for pkg_desc_str in output_str.split('Package: '):
            if not pkg_desc_str:
                continue

            m = re.search(r'^Version:\s*(\S+)\s*$', pkg_desc_str, flags=re.M)
            if not m:
                raise SandboxTaskFailureError('Cannot parse description for package {0}'.format(pkg_name))
            ver = m.group(1)

            m = re.search(r'^Archive:(.*)$', pkg_desc_str, flags=re.M)
            if not m:
                raise SandboxTaskFailureError('Cannot parse description for package {0}'.format(pkg_name))

            repos = [s.strip() for s in m.group(1).split(',')]
            if Repository.STABLE in repos:
                repo = Repository.STABLE
            elif Repository.TESTING in repos:
                repo = Repository.TESTING
            elif Repository.UNSTABLE in repos:
                repo = Repository.UNSTABLE
            else:
                repo = m.group(1)

            pkg_info = PackageInfo(pkg_name, ver, repo)
            pkg_info_list.append(pkg_info)

            def parse_package_dep_str(list_str):
                ret = []
                pkg_regex = re.compile(r'(\S+)(\s*\((\S+)\s+(\S+)\))?')
                for s in list_str.split(','):
                    m = pkg_regex.match(s.strip())
                    if not m:
                        raise SandboxTaskFailureError('Cannot parse package list element: {0}'.format(s.strip()))
                    d = PackageDependency()
                    d.name = m.group(1)
                    d.op = m.group(3)
                    d.version = m.group(4)
                    ret.append(d)
                return ret

            m = re.search(r'^Depends:(.*$\n(?:^\s.*$\n)*)', pkg_desc_str, flags=re.M)
            if m:
                pkg_info.dependencies = parse_package_dep_str(m.group(1))

            m = re.search(r'^Conflicts:(.*$\n(?:^\s.*$\n)*)', pkg_desc_str, flags=re.M)
            if m:
                pkg_info.conflicts = parse_package_dep_str(m.group(1))

        return pkg_info_list if not version else pkg_info_list[0]

    def version_to_tuple_maps(self, ver):
        m = re.match(r'(\d+)(?:\.(\d+)(?:\.(\d+)(?:-(.*\.)?(\d+)(.*))?)?)?$', ver)

        if not m:
            return

        major = int(m.group(1) or 0)
        minor = int(m.group(2) or 0)
        fix = int(m.group(3) or 0)
        misc1 = m.group(4) or ''
        build = int(m.group(5) or 0)
        misc2 = m.group(6) or ''

        return (major, minor, fix, misc1, build, misc2)

    def version_to_tuple_p2p(self, ver):
        m = re.match(r'(\d+)(?:\.(\d+)(?:-(\d+))?)?$', ver)

        if not m:
            return

        major = int(m.group(1) or 0)
        minor = int(m.group(2) or 0)
        fix = int(m.group(3) or 0)
        misc1 = ''
        build = 0
        misc2 = ''

        return (major, minor, fix, misc1, build, misc2)

    def version_to_tuple(self, pkg_name, ver):
        tuple = self.version_to_tuple_maps(ver)
        if not tuple:
            tuple = self.version_to_tuple_p2p(ver)
        if not tuple:
            raise SandboxTaskFailureError('Wrong version of {pkg}: {ver}'.format(
                pkg=pkg_name,
                ver=ver
            ))

        return tuple

    def check_version_constraint(self, pkg_name, ver, constraint_op, constraint_ver):
        ver_tuple = self.version_to_tuple(pkg_name, ver)
        constraint_ver_tuple = self.version_to_tuple(pkg_name, constraint_ver)

        if constraint_op == '=':
            return ver_tuple == constraint_ver_tuple
        elif constraint_op == '<':
            return ver_tuple < constraint_ver_tuple
        elif constraint_op == '<=':
            return ver_tuple <= constraint_ver_tuple
        elif constraint_op == '>':
            return ver_tuple > constraint_ver_tuple
        elif constraint_op == '>=':
            return ver_tuple >= constraint_ver_tuple

        raise SandboxTaskFailureError('Unknown operator: %s' % constraint_op)

    def get_packages_list(self, stub_set, desired_packages, log_file=None):
        # Собираем инфу про все пакеты с зависимостями
        pkg_infos = {}
        to_process = list(desired_packages)
        processed = set([p.name for p in desired_packages])
        while len(to_process):
            # Получаем информацию о пакете
            desired_pkg = to_process.pop(0)
            if desired_pkg.version:
                pkg_info = self.get_package_info(desired_pkg.name, version=desired_pkg.version, log_file=log_file)
                if not pkg_info:
                    raise SandboxTaskUnknownError('Cannot get info for %s=%s' % (desired_pkg.name, desired_pkg.version))
            else:
                pkg_info = self.choose_right_package_version(
                    desired_pkg.name, desired_pkg.repository, log_file=log_file
                )
                if not pkg_info:
                    raise SandboxTaskUnknownError('Cannot choose right version for %s' % desired_pkg.name)

            # Виртуальные пакеты должны быть удовлетворены ранее
            if pkg_info.virtual():
                if pkg_info.name in stub_set:
                    continue

                provided = False
                for p in pkg_info.provided_by:
                    if p.name in pkg_infos and pkg_infos[p.name].version == p.version:
                        provided = True
                        break
                if not provided:
                    raise SandboxTaskFailureError('Virtual package {0} not provided by anyone'.format(pkg_info.name))
            else:
                pkg_infos[pkg_info.name] = pkg_info
                for d in pkg_info.dependencies:
                    if 'yandex' not in d.name:
                        continue
                    if d.name in stub_set:
                        continue
                    if d.name in processed:
                        continue

                    logging.info('From package {0} add dependency {1} '.format(pkg_info.name, d.name))

                    processed.add(d.name)
                    to_process.append(PackageDescr(d.name, None, desired_pkg.repository))

        # Проверяем отсутствие конфликтов
        logging.debug('Checking conflicts')
        for pkg_info in pkg_infos.values():
            logging.debug('Package: %s' % pkg_info.name)
            for d in pkg_info.dependencies:
                logging.debug('Dependency: %s %s %s' % (d.name, d.op, d.version))
                dep_info = pkg_infos.get(d.name)
                if not dep_info:
                    continue
                if not d.op:
                    continue
                if not self.check_version_constraint(dep_info.name, dep_info.version, d.op, d.version):
                    raise SandboxTaskFailureError(
                        'Conflict: {pkg}={pkg_ver} need {dep}{op}{constr_ver} '
                        'but {inst_ver} to be installed'.format(
                            pkg=pkg_info.name,
                            pkg_ver=pkg_info.version,
                            dep=dep_info.name,
                            op=d.op,
                            constr_ver=d.version,
                            inst_ver=dep_info.version
                        ))

            for c in pkg_info.conflicts:
                logging.debug('Conflict: %s %s %s' % (c.name, c.op, c.version))
                conf_info = pkg_infos.get(c.name)
                if not conf_info:
                    continue
                if not c.op or self.check_version_constraint(conf_info.name, conf_info.version, c.op, c.version):
                    raise SandboxTaskFailureError(
                        'Conflict: {pkg}={pkg_ver} conflicts with {conf}{op}{constr_ver} '
                        'but {inst_ver} to be installed'.format(
                            pkg=pkg_info.name,
                            pkg_ver=pkg_info.version,
                            conf=conf_info.name,
                            op=c.op,
                            constr_ver=c.version,
                            inst_ver=conf_info.version
                        ))

        # Выстраиваем в нужном порядке установки
        # Простой обход в глубину
        processed_pkgs = set()
        install_order = []
        for p in desired_packages:
            if p.name in processed_pkgs:
                continue
            path = [(p.name, 0)]
            processed_pkgs.add(p.name)
            while len(path):
                pkg_name, depidx = path[-1]
                pkg_info = pkg_infos[pkg_name]
                if len(pkg_info.dependencies) == depidx:
                    install_order.append(pkg_info)
                    path.pop(-1)
                    continue
                path[-1] = (pkg_name, depidx + 1)
                dep = pkg_info.dependencies[depidx]
                if dep.name in processed_pkgs:
                    continue
                if dep.name not in pkg_infos:
                    continue
                path.append((dep.name, 0))
                processed_pkgs.add(dep.name)

        if len(processed_pkgs) != len(pkg_infos.keys()):
            raise SandboxTaskUnknownError('Processed not all packages %s' % str(processed_pkgs))

        return ['{0}={1}'.format(p.name, p.version) for p in install_order]

    def on_execute(self):
        out_resource = channel.sandbox.get_resource(self.ctx['out_resource_id'])

        res = []
        for cmd in self.ctx[PreInstallCommandsParameter.name].split():
            cmd.strip()
            res.append('cmd: {0}'.format(cmd))

        stub_set = set()
        for pkg in self.ctx[StubbedPackagesParameter.name].split():
            pkg.strip()
            stub_set.add(pkg)
            res.append('stub: {0}'.format(pkg))

        desired_packages = []

        def make_desired_packages(list_str, repo):
            ret = []
            for p in list_str.split():
                pkg_extened_name = p.strip()
                i = pkg_extened_name.find('=')
                pkg_descr = PackageDescr(repository=repo)
                if i > 0:
                    pkg_descr.name, pkg_descr.version = pkg_extened_name.split('=')[:2]
                else:
                    pkg_descr.name = pkg_extened_name
                ret.append(pkg_descr)
            return ret

        desired_packages += make_desired_packages(self.ctx[ManualPackageListStableParameter.name],
                                                  Repository.STABLE)
        desired_packages += make_desired_packages(self.ctx[ManualPackageListTestingParameter.name],
                                                  Repository.TESTING)
        desired_packages += make_desired_packages(self.ctx[ManualPackageListUnstableParameter.name],
                                                  Repository.UNSTABLE)

        log_filename = get_unique_file_name(self.log_path(), 'executing.log')
        log_file = open(log_filename, 'w')
        logging.info('Writing all exec info into {0}'.format(log_file))
        with closing(log_file):
            self.exec_cmd('sudo apt-get update', log_file=log_file)
            self.exec_cmd('sudo apt-get install -yf aptitude', log_file=log_file)
            res += self.get_packages_list(stub_set, desired_packages, log_file)

        out_file = open(out_resource.path, 'w')
        out_file.write('\n'.join(res) + '\n')
        out_file.close()

        resource_attributes = self.ctx.get(SetAttrsParameter.name)
        if resource_attributes:
            logging.info('Set resource attributes %s', resource_attributes)
            for k, v in string.parse_attrs(resource_attributes).iteritems():
                channel.sandbox.set_resource_attribute(self.ctx['out_resource_id'], k, v)


__Task__ = MakeMapsPackagesList
