import os
import glob
import logging
import shutil
import subprocess
import tarfile

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import ListRepeater
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk import environments

from sandbox.projects import resource_types

from sandbox.projects.common.nanny import nanny


class PERSONAL_SEARCH_DAEMON_RELEASABLE(resource_types.AbstractResource):
    """
        Personal search services daemons bundles (tar.gz), releasable
    """
    releasable = True
    executable = False
    auto_backup = True


class PERSONAL_SEARCH_DAEMON_IEX_PROXY(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_MSEARCH_PROXY(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_LUCENE(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_PERSEUS(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_ZOOLOSER(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_YADISK_SEARCH_PROXY(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_SALO(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_TUPITA(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_LOGBROKER_CONSUMER(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_LUCENE_MAIL_CONFIG(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class PERSONAL_SEARCH_DAEMON_ZORA_PROXY(PERSONAL_SEARCH_DAEMON_RELEASABLE):
    pass


class RepoUrl(SandboxStringParameter):
    name = 'repo'
    description = 'Url to git repo'
    default_value = 'https://git.yandex.ru/saas/personal.git'
    required = True


class ProjectNames(ListRepeater, SandboxStringParameter):
    name = 'project_names'
    description = 'Project names to make bundles for'
    required = True


class JavaResourceId(SandboxStringParameter):
    name = 'java_resource_id'
    description = 'Sandbox resource id with jdk'
    required = True


class RagelResourceId(SandboxStringParameter):
    name = 'ragel_resource_id'
    description = 'Sandbox resource id with ragel'
    required = True


class ZlibsResourceId(SandboxStringParameter):
    name = 'zlibs_resource_id'
    description = 'Sandbox resource id with zlibs'
    required = True


class SqliteResourceId(SandboxStringParameter):
    name = 'sqlite_resource_id'
    description = 'Sandbox resource id with sqlite'
    required = True


class GccResourceId(SandboxStringParameter):
    name = 'gcc_resource_id'
    description = 'Sandbox resource id with gcc'
    required = True


class GenerateSonarReport(SandboxBoolParameter):
    name = 'generate_sonar_report'
    description = "Set to True if you want to run 'sonar-scanner'"
    required = False
    default_value = False


class SonarScannerResourceId(SandboxStringParameter):
    name = 'sonar-scanner_resource_id'
    description = 'Sandbox resource id with sonar scanner archive'
    required = False


class NumberOfTries(SandboxIntegerParameter):
    name = 'number_of_tries'
    description = 'Number of tries for make command'
    required = False
    default_value = 15


class NumberOfThreads(SandboxIntegerParameter):
    name = 'number_of_threads'
    description = 'Number of threads for make command'
    required = False
    default_value = 4


class AddAllToMake(SandboxBoolParameter):
    name = 'add_all_to_make'
    description = "Set to True if you want to run 'make' command with 'all'"
    required = False
    default_value = False


class MakePersonalBundle(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Makes sandbox resource with bundle for specified project
        from 'personal' git repo
    """

    type = 'MAKE_PERSONAL_BUNDLE'

    execution_space = 15000
    environment = (environments.PipEnvironment('GitPython'),)

    input_parameters = [
        RepoUrl,
        ProjectNames,
        JavaResourceId,
        RagelResourceId,
        ZlibsResourceId,
        SqliteResourceId,
        GccResourceId,
        GenerateSonarReport,
        SonarScannerResourceId,
        NumberOfTries,
        NumberOfThreads,
        AddAllToMake
    ]

    def on_execute(self):
        import git

        projects_to_resources = {}
        projects_to_resources['iex-proxy'] = PERSONAL_SEARCH_DAEMON_IEX_PROXY
        projects_to_resources['msearch-proxy'] = PERSONAL_SEARCH_DAEMON_MSEARCH_PROXY
        projects_to_resources['lucene'] = PERSONAL_SEARCH_DAEMON_LUCENE
        projects_to_resources['perseus'] = PERSONAL_SEARCH_DAEMON_PERSEUS
        projects_to_resources['zooloser'] = PERSONAL_SEARCH_DAEMON_ZOOLOSER
        projects_to_resources['yadisk-search-proxy'] = PERSONAL_SEARCH_DAEMON_YADISK_SEARCH_PROXY
        projects_to_resources['salo'] = PERSONAL_SEARCH_DAEMON_SALO
        projects_to_resources['tupita'] = PERSONAL_SEARCH_DAEMON_TUPITA
        projects_to_resources['logbroker-consumer'] = PERSONAL_SEARCH_DAEMON_LOGBROKER_CONSUMER
        projects_to_resources['lucene-mail-config'] = PERSONAL_SEARCH_DAEMON_LUCENE_MAIL_CONFIG
        projects_to_resources['zora-proxy'] = PERSONAL_SEARCH_DAEMON_ZORA_PROXY

        task_dir = os.getcwd()
        java_resource_id = self.ctx.get(JavaResourceId.name)
        logging.info("java resource id is " + java_resource_id)
        java_resource = channel.sandbox.get_resource(java_resource_id)
        if java_resource is not None:
            logging.info("java resource is not None")
            java_resource_path = self.sync_resource(java_resource)
            logging.info("java resource path === " + java_resource_path)
            tarfile.open(java_resource_path).extractall('java_resource')
            logging.info("Opened java_resource file")
        else:
            logging.info("java_resource is None")

        ragel_resource_id = self.ctx.get(RagelResourceId.name)
        logging.info("ragel resource id is " + ragel_resource_id)
        ragel_resource = channel.sandbox.get_resource(ragel_resource_id)
        if ragel_resource is not None:
            logging.info("ragel resource is not None")
            ragel_resource_path = self.sync_resource(ragel_resource)
            logging.info("ragel_res_path = " + ragel_resource_path)
            self._subprocess(
                "tar -xf " + ragel_resource_path + " -C " + task_dir,
                wait=True,
                shell=True,
                out_file_name="ragel.out",
                log_prefix="ragel"
            )
            logging.info("Opened ragel_resource file")
        else:
            logging.info("ragel_resource is None")

        out_files_copy_dir = "out_files_copy"

        self._subprocess(
            "mkdir " + os.path.join(self.log_path(), out_files_copy_dir),
            wait=True,
            shell=True,
            out_file_name="mkdir_for_out_files",
            log_prefix="mkdir"
        )

        gcc_resource_id = self.ctx.get(GccResourceId.name)
        logging.info("gcc resource id is " + gcc_resource_id)
        gcc_resource = channel.sandbox.get_resource(gcc_resource_id)
        if gcc_resource is not None:
            logging.info("gcc resource is not None")
            gcc_resource_path = self.sync_resource(gcc_resource)
            logging.info("gcc_res_path = " + gcc_resource_path)
            self._subprocess(
                "tar -xf " + gcc_resource_path + " -C " + task_dir,
                wait=True,
                shell=True,
                out_file_name="gcc.out",
                log_prefix="gcc"
            )
            logging.info("Opened gcc_resource file")
        else:
            logging.info("gcc_resource is None")

        if self.ctx.get(GenerateSonarReport.name):
            sonar_resource_id = self.ctx.get(SonarScannerResourceId.name)
            logging.info("sonar resource id is " + sonar_resource_id)
            sonar_resource = channel.sandbox.get_resource(sonar_resource_id)
            if sonar_resource is not None:
                logging.info("sonar resource is not None")
                sonar_resource_path = self.sync_resource(sonar_resource)
                logging.info("sonar_res_path = " + sonar_resource_path)
                self._subprocess(
                    "unzip " + sonar_resource_path,
                    wait=True,
                    shell=True,
                    out_file_name="sonar.out",
                    log_prefix="sonar"
                )
                logging.info("Opened sonar_resource file")
            else:
                logging.info("sonar_resource is None")

        for filename in glob.glob(os.path.join(os.getcwd(), '*.out')):
            logging.info("copying " + str(filename) + " to " + os.path.join(self.log_path(), out_files_copy_dir))
            shutil.copy(filename, os.path.join(self.log_path(), out_files_copy_dir))

        logging.info("Cloning repo...")
        # git.Git().clone(self.ctx[RepoUrl.name], 'personal')
        git_out = open('git.out', 'w')
        res = subprocess.call(['git', 'clone', self.ctx[RepoUrl.name], 'personal', '-c', 'http.postBuffer=1048576000', '--progress'], stdout=git_out, stderr=subprocess.STDOUT)
        git_out.close()

        if res != 0:
            raise Exception('Failed to clone git repo ' + self.ctx[RepoUrl.name])

        os.chdir('personal')
        repo = git.Repo('.')

        logging.info("cloned")
        g = git.cmd.Git()
        headcommit = repo.head.commit
        logging.info("message = " + str(headcommit.message))
        logging.info("sha1 = " + str(headcommit.hexsha))
        logging.info("author = " + str(headcommit.author))
        author_email = g.log("-n1 --pretty=format:'%aE'")
        logging.info("author email = " + author_email)

        first_line = g.log("--abbrev-commit").split('\n')[0]
        logging.info("first line = " + first_line)
        abbrev_commit = first_line.split()[1]
        logging.info('abbrev_commit = ' + abbrev_commit)

        zlibs_resource_id = self.ctx.get(ZlibsResourceId.name)
        logging.info("zlibs resource id is " + zlibs_resource_id)
        zlibs_resource = channel.sandbox.get_resource(zlibs_resource_id)
        if zlibs_resource is not None:
            logging.info("zlibs resource is not None")
            zlibs_resource_path = self.sync_resource(zlibs_resource)
            logging.info("zlibs_res_path = " + zlibs_resource_path)
            self._subprocess(
                "tar -xf " + zlibs_resource_path + " -C " + os.getcwd(),
                wait=True,
                shell=True,
                out_file_name="zlibs.out",
                log_prefix="zlibs"
            )
            logging.info("Opened zlibs_resource file")
        else:
            logging.info("zlibs_resource is None")

        os.chdir('build')
        sqlite_resource_id = self.ctx.get(SqliteResourceId.name)
        logging.info("sqlite resource id is " + sqlite_resource_id)
        sqlite_resource = channel.sandbox.get_resource(sqlite_resource_id)
        if sqlite_resource is not None:
            logging.info("sqlite resource is not None")
            sqlite_resource_path = self.sync_resource(sqlite_resource)
            logging.info("sqlite_res_path = " + sqlite_resource_path)
            self._subprocess(
                "cp " + sqlite_resource_path + " " + os.getcwd(),
                wait=True,
                shell=True,
                out_file_name="sqlite.out",
                log_prefix="sqlite"
            )
            logging.info("Copied sqlite.tar.gz")
        else:
            logging.info("sqlite_resource is None")

        os.chdir(os.path.dirname(os.getcwd()))

        projects_string = ""
        for project_name in self.ctx.get(ProjectNames.name):
            projects_string += " " + project_name + "-bundle"

        if self.ctx.get(AddAllToMake.name):
            projects_string += " all"

        logging.info("projects_string = " + projects_string)

        make_bundle_iex_command = \
            "make -j " + str(self.ctx.get(NumberOfThreads.name)) + " CC=$PWD/../gcc/bin/gcc-6.1 JAVA_HOME=$PWD/../java_resource " + \
            "PATH=$PATH:$PWD/../usr/bin ALLOW_CLONE=false DISPLAY=" + projects_string
        self.set_info("Making projects...")
        done = False
        tries = self.ctx.get(NumberOfTries.name)
        MAIL_TO = "atugaenko"
        subject = "report from sandbox"
        text = "last committer to git is " + str(headcommit.author) + ' ' + str(headcommit.hexsha)
        for i in range(tries):
            if not done:
                try:
                    self._subprocess(
                        make_bundle_iex_command,
                        wait=True,
                        shell=True,
                        out_file_name="bundle.out",
                        log_prefix='bundle'
                    )
                    done = True
                except Exception as e:
                    for filename in glob.glob(os.path.join(os.getcwd(), 'bundle.*')):
                        shutil.copy(filename, os.path.join(self.log_path(), out_files_copy_dir))
                    logging.error('error in make command, try ' + str(i+1) + ' got: ' + str(e))
        if not done:
            try:
                make_bundle_iex_command_one_thread = \
                    "make CC=$PWD/../gcc/bin/gcc-6.1 JAVA_HOME=$PWD/../java_resource " + \
                    "PATH=$PATH:$PWD/../usr/bin ALLOW_CLONE=false DISPLAY=" + projects_string
                self._subprocess(
                    make_bundle_iex_command_one_thread,
                    wait=True,
                    shell=True,
                    out_file_name="bundle_one_thread.out",
                    log_prefix='bundle_one_thread'
                )
                done = True
            except Exception as e:
                for filename in glob.glob(os.path.join(os.getcwd(), 'bundle.*')):
                    shutil.copy(filename, os.path.join(self.log_path(), out_files_copy_dir))
                logging.error('error in make command in one thread, got: ' + str(e))
        if not done:  # if still not done after one-thread execution
            logging.info("Error while making projects")
            text += ', error while making projects'
            channel.sandbox.send_email(
                mail_to=MAIL_TO,
                mail_cc='',
                mail_subject=subject,
                mail_body=text)
            self._subprocess(
                (
                    "mkdir -p bundle_failures_dir "
                    "&& find build/report -name '*.txt.try' | xargs cp -t bundle_failures_dir "
                    "&& tar czf bundle.failures.tar.gz bundle_failures_dir"
                ),
                wait=True,
                shell=True,
                out_file_name="bundle_failures.out",
                log_prefix="bundle_failures"
            )
            for filename in glob.glob(os.path.join(os.getcwd(), 'bundle.*')):
                shutil.copy(filename, os.path.join(self.log_path(), out_files_copy_dir))
            raise Exception("can't make the resource")

        for filename in glob.glob(os.path.join(os.getcwd(), 'bundle.*')):
            shutil.copy(filename, os.path.join(self.log_path(), out_files_copy_dir))

        logging.info("Done")
        logging.info("Sending notification about last commit")
        channel.sandbox.send_email(
            mail_to=MAIL_TO,
            mail_cc='',
            mail_subject=subject,
            mail_body=text)

        if self.ctx.get(GenerateSonarReport.name):
            self._subprocess(
                "make build/sonar.config",
                wait=True,
                shell=True,
                out_file_name="make_sonar_config.out",
                log_prefix='make_sonar_config'
            )
            self._subprocess(
                'SONAR_SCANNER_OPTS=-Dsonar.host.url=http://sonar.qatools.yandex-team.ru:80\\\\ -Xmx8G JAVA_HOME=$PWD/../java_resource ./../sonar-scanner-3.0.3.778-linux/bin/sonar-scanner',
                wait=True,
                shell=True,
                out_file_name="sonar_scanner.out",
                log_prefix="sonar_scanner"
            )

        for project_name in self.ctx.get(ProjectNames.name):

            self.set_info(
                'Creating sandbox resourse for ' + project_name + ' project...'
            )

            bundle_path = os.path.join(
                os.getcwd(),
                "build",
                "bundle",
                project_name + "-bundle.tar.gz"
            )
            logging.info("path to bundle = " + bundle_path)

            self.create_resource(
                description=(
                    'Bundle for ' +
                    project_name + '-' + abbrev_commit + '\n' +
                    'commit ' + headcommit.hexsha + '\n' +
                    headcommit.message
                ),
                resource_path=bundle_path,
                resource_type=projects_to_resources[project_name],
                arch='any',
                attributes=dict(
                    ttl='inf',
                )
            )
            self.set_info('Done.')

        self.set_info('Resources for all projects are done.')


__Task__ = MakePersonalBundle
