# coding: utf-8

import os
import time
import logging

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import add_write_permissions_for_path
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.utils import get_bsconfig
from sandbox.sandboxsdk.channel import channel


class ResourcesList(SandboxStringParameter):
    name = 'resources_list'
    description = 'Resources ids list (ex.: 1234,31231,123412)'
    default_value = ''
    required = True


class ShardPrefix(SandboxStringParameter):
    name = 'shard_prefix'
    description = 'Shard name prefix:'
    default_value = 'default_shard'
    required = True


class ResultResourceType(SandboxStringParameter):
    name = 'result_resource_type'
    description = 'Type name of the result resource:'
    default_value = str(resource_types.OTHER_RESOURCE)

    choices = [(str(i), str(i)) for i in resource_types.AbstractResource if 'SHARD' in str(i).upper()]


class MakeShard(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Создать шард из списка переданных ресурсов.

        Просто создаёт папку, копирует туда ресурсы, потом делает shard_init
    """
    type = 'MAKE_SHARD'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [
        ResourcesList,
        ShardPrefix,
        ResultResourceType
    ]

    def arcadia_info(self):
        """
            Получение информации о задаче при релизе

            :return список из трёх значений revision, tag, branch
        """
        return None, self.ctx.get('shard_name', 'unknown shard'), None

    def on_execute(self):
        resources = [r.strip() for r in str(self.ctx.get('resources_list', '')).split(',')]
        if not resources:
            raise errors.SandboxTaskFailureError("Resources list is empty")

        # проверяем ресурсы
        resources_objects = []
        for resource_id in resources:
            resource = channel.sandbox.get_resource(resource_id)
            if resource:
                resources_objects.append(resource)
            else:
                raise errors.SandboxTaskFailureError("Cannot get {} resource".format(resource_id))

        # создаём директорию шарда

        shard_name = '{}-000-{}'.format(self.ctx['shard_prefix'], int(time.time()))
        self.ctx['shard_name'] = shard_name
        logging.info('Make shard with name {}'.format(shard_name))
        shard_folder = self.path(shard_name)
        make_folder(shard_folder, delete_content=True)

        # копируем всё в папку шарда
        for resource in resources_objects:
            logging.info('Copy resource {}'.format(resource.id))
            resource_path = self.sync_resource(resource.id)
            resource_basename = os.path.basename(resource_path)
            new_path = os.path.join(shard_folder, resource_basename)
            copy_path(resource_path, os.path.join(shard_folder, resource_basename))
            add_write_permissions_for_path(new_path)

        # создаём ресурс и делаем shard_init
        try:
            self.create_resource(
                description='shard {}'.format(shard_name),
                resource_path=shard_folder,
                resource_type=self.ctx['result_resource_type'],
                arch='linux'
            )
        except ValueError:
            raise SandboxTaskFailureError('Unknown resource type {}'.format(self.ctx['result_resource_type']))
        # do initialize shard
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config"
        )


__Task__ = MakeShard
