# coding=utf-8

import sys
import logging

import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters

import collections


class OperationBlock(object):
    def __init__(self, block_info):
        print '1'
        assert(block_info['type'] == 'operation')
        self.block_code = block_info['blockCode']
        # self.block_guid = block_info['blockGuid']
        self.name = block_info['name']
        self.op_id = block_info['operationId']

    def AddParameters(self, api):
        black_list = set(
            [
                'job-command',
                'job-binary-url',
                'slave-job-command',
                'slave-job-binary-url',
                'master-job-command',
                'master-job-binary-url',
            ])
        self.params = collections.OrderedDict()
        rslt = api.GetBlockParametersByCode(self.block_code)['result'][0]['parameters']
        for param in sorted(rslt):
            if 'value' in param:
                param_type = type(param['value'])
                value = param['value']
                name = param['parameter']
                if name not in black_list:
                    self.params[name] = {
                        'value': value,
                        'type': repr(param_type)
                    }
        pass

    def GetCode(self):
        return self.block_code

    def GetJSON(self):
        rslt = collections.OrderedDict()
        rslt['block_code'] = self.block_code
        rslt['name'] = self.name
        rslt['op_id'] = self.op_id
        rslt['params'] = self.params
        return rslt


class OperationConnection(object):
    def __init__(self, conn_info):
        assert(conn_info['@type'] == 'OperationConnectionInfo')
        self.dest_block_code = conn_info['destBlockCode']
        self.dest_endpiont_name = conn_info['destEndpointName']
        self.src_block_code = conn_info['sourceBlockCode']
        self.src_endpiont_name = conn_info['sourceEndpointName']

    def GetJSON(self):
        rslt = collections.OrderedDict()
        rslt['dest_block_code'] = self.dest_block_code
        rslt['dest_endpiont_name'] = self.dest_endpiont_name
        rslt['src_block_code'] = self.src_block_code
        rslt['src_endpiont_name'] = self.src_endpiont_name
        return rslt


class NirvanaDumper(object):
    def __init__(self, wfid, nv_token, algorc_nirvana_api):
        self.wfid = wfid
        self.nv_token = nv_token
        self.api = algorc_nirvana_api.NirvanaAPI(self.wfid, self.nv_token)

        self.Parse()

    def Parse(self):
        rslt = self.api.GetWorkflow()
        self.op_blocks = {}
        for block_info in rslt['result']['blocks']:
            if block_info['type'] == 'operation':
                op_block = OperationBlock(block_info)
                op_block.AddParameters(self.api)
                self.op_blocks[op_block.GetCode()] = op_block
            else:
                print block_info

        self.conns_info = []
        for conn_info in rslt['result']['connections']:
            if conn_info['@type'] == 'OperationConnectionInfo':
                op_conn = OperationConnection(conn_info)
                self.conns_info.append(op_conn)
            else:
                print conn_info

    def Dump(self):
        json_dump = collections.OrderedDict()
        json_dump['blocks'] = {}
        for block_code in self.op_blocks:
            json_dump['blocks'][block_code] = self.op_blocks[block_code].GetJSON()
        json_dump['connectons'] = []
        for conn in self.conns_info:
            json_dump['connectons'].append(conn.GetJSON())
        return json_dump


class Operation(object):
    def __init__(self, op_id, api, name):
        self.api = api
        self.op_id = op_id
        self.name = name

    def Create(self):
        self.new_op_id = self.api.CloneOperation(self.op_id, self.name)['result']

    def GetNewId(self):
        return self.new_op_id

    def UpdateJobBinaryUrl(self, stored_data_guid):
        self._UpdateJobBinaryUrl(stored_data_guid)

    def UpdateVersion(self, new_version):
        self.api.EditOperationMeta(self.new_op_id, version_number=new_version)

    def Approve(self):
        self.api.ApproveOperation(self.new_op_id)


class MSOperation(Operation):
    def __init__(self, op_id, api, name):
        self.api = api
        self.op_id = op_id
        self.name = name

    def _UpdateJobBinaryUrl(self, stored_data_guid):
        rslt = self.api.EditOperationParameters(self.new_op_id, 'master-job-binary-url', new_default_value=stored_data_guid)
        logging.info(rslt)
        rslt = self.api.EditOperationParameters(self.new_op_id, 'slave-job-binary-url', new_default_value=stored_data_guid)
        logging.info(rslt)


class LocalOperation(Operation):
    def __init__(self, op_id, api, name):
        self.api = api
        self.op_id = op_id
        self.name = name

    def _UpdateJobBinaryUrl(self, stored_data_guid):
        rslt = self.api.EditOperationParameters(self.new_op_id, 'job-binary-url', new_default_value=stored_data_guid)
        logging.info(rslt)


class OperationCreator(object):
    def __init__(self, nirvana_api, token, storagePath, mr_local_orig_op, mr_ms_orig_op, tsv_local_orig_op, tsv_ms_orig_op, new_version):
        self.storagePath = storagePath
        self.api = nirvana_api.NirvanaAPI('foo', token)

        self.mr_local = LocalOperation(mr_local_orig_op, self.api, 'Train matrixnet on CPU (from MR, single node) [official from ML team][Testing]')
        self.mr_ms = MSOperation(mr_ms_orig_op, self.api, 'Train matrixnet on CPU (from MR) [official from ML team][Testing]')

        self.tsv_local = LocalOperation(tsv_local_orig_op, self.api, 'Train matrixnet on CPU (single node) [official from ML team][Testing]')
        self.tsv_ms = MSOperation(tsv_ms_orig_op, self.api, 'Train matrixnet on CPU [official from ML team][Testing]')
        self.new_version = new_version

    def GetMrLocalOperation(self):
        return self.mr_local

    def GetMrMsOperation(self):
        return self.mr_ms

    def GetTsvLocalOperation(self):
        return self.tsv_local

    def GetTsvMsOperation(self):
        return self.tsv_ms

    def Create(self):
        self.mr_local.Create()
        self.mr_local.UpdateJobBinaryUrl(self.storagePath)
        self.mr_local.UpdateVersion(self.new_version)
        self.mr_local.Approve()

        self.mr_ms.Create()
        self.mr_ms.UpdateJobBinaryUrl(self.storagePath)
        self.mr_ms.UpdateVersion(self.new_version)
        self.mr_ms.Approve()

        self.tsv_local.Create()
        self.tsv_local.UpdateJobBinaryUrl(self.storagePath)
        self.tsv_local.UpdateVersion(self.new_version)
        self.tsv_local.Approve()

        self.tsv_ms.Create()
        self.tsv_ms.UpdateJobBinaryUrl(self.storagePath)
        self.tsv_ms.UpdateVersion(self.new_version)
        self.tsv_ms.Approve()


class Task(sdk_task.SandboxTask):
    """ Create a Train matrixnet on CPU Nirvana operations. """

    class TokenName(sdk_parameters.SandboxStringParameter):
        name = "nirvana_token"
        description = "Vault with Nirvana Token"

    class StTokenName(sdk_parameters.SandboxStringParameter):
        name = "startrek_token"
        description = "Vault with Startrek Token"

    class StoredDataGuid(sdk_parameters.SandboxStringParameter):
        name = "stored_data_guid"
        description = "stored data guid of job-binary-url package"

    class StoragePath(sdk_parameters.SandboxStringParameter):
        name = "storagePath"
        description = "storage path of job-binary-url package"

    class MrLocalOriginalOpId(sdk_parameters.SandboxStringParameter):
        name = "mr_local_orig_op_id"
        description = "the original MR LOCAL Nirvana Oeration id"

    class MrMsOriginalOpId(sdk_parameters.SandboxStringParameter):
        name = "mr_ms_orig_op_id"
        description = "the original MR MS Nirvana Oeration id"

    class TsvLocalOriginalOpId(sdk_parameters.SandboxStringParameter):
        name = "Tsv_local_orig_op_id"
        description = "the original TSV LOCAL Nirvana Oeration id"

    class TsvMsOriginalOpId(sdk_parameters.SandboxStringParameter):
        name = "tsv_ms_orig_op_id"
        description = "the original TSV MS Nirvana Oeration id"

    class NewVersion(sdk_parameters.SandboxStringParameter):
        name = "new_version"
        description = "New version of the operations"

    class MatrinxetRevision(sdk_parameters.SandboxStringParameter):
        name = "matrixnet_revision"
        description = "the revision of matrixnet"

    class RunMatrinxetRevision(sdk_parameters.SandboxStringParameter):
        name = "run_matrixnet_revision"
        description = "the revision of run_matrixnet scriots"

    class StartrekTicket(sdk_parameters.SandboxStringParameter):
        name = "startrek_ticket"
        description = "startrek ticket for report generation"

    type = "MAKE_TRAIN_MATRIXNET_ON_CPU_NIRVANA_OPS"
    input_parameters = [TokenName, StTokenName, StoredDataGuid, StoragePath,
                        MrLocalOriginalOpId, MrMsOriginalOpId, TsvLocalOriginalOpId, TsvMsOriginalOpId,
                        NewVersion, MatrinxetRevision, RunMatrinxetRevision, StartrekTicket]

    def MakeStartrekCommentWithOperationsTable(
        self,
        startrek_api_token, ticket_id,
        new_mr_ms_op_id, new_mr_local_op_id, new_tsv_ms_op_id, new_tsv_local_op_id,
        matrixnet_rev, run_matrixnet_rev, new_version
    ):
        import startrack_api
        import get_file_strategy
        import get_file

        operations_table = """#|
    ||{ver} | MR | TSV||
    || MS| (({mr_ms} mr_ms))| (({tsv_ms} tsv_ms))||
    || LOCAL| (({mr_local} mr_local)) | (({tsv_local} tsv_local))||
    |#
matrixnet binary revision: {mx_rev}
run_matrixnet.py script revision: {run_mx_rev}

"""

        def make_url_for_operation(opid):
            return "https://nirvana.yandex-team.ru/operation/{0}/overview".format(opid)

        op_data = {'mr_ms': make_url_for_operation(new_mr_ms_op_id),
                   'mr_local': make_url_for_operation(new_mr_local_op_id),
                   'tsv_ms': make_url_for_operation(new_tsv_ms_op_id),
                   'tsv_local': make_url_for_operation(new_tsv_local_op_id),
                   'mx_rev': matrixnet_rev,
                   'run_mx_rev': run_matrixnet_rev,
                   'ver': new_version
                   }

        logging.info(operations_table.format(**op_data))

        comment_helper = startrack_api.Comments(ticket_id, startrek_api_token, get_file_strategy=get_file_strategy.return_get_file_class(get_file.GetFile))
        logging.info(comment_helper.POST(operations_table.format(**op_data)))

    def on_execute(self):
        sys.path.append(sdk_svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/quality/relev_tools/conveyor_dashboard/commons"))
        import nirvana_api

        # sys.path.append(sdk_svn.Arcadia.get_arcadia_src_dir("arcadia:/arc/trunk/arcadia/quality/nirvana_tools/workflow_dump_to_json"))
        # import workflow_dump_to_json

        token = self.get_vault_data('algorc_nirvana_token')

        mr_local_orig_op = self.ctx[self.MrLocalOriginalOpId.name]
        mr_ms_orig_op = self.ctx[self.MrMsOriginalOpId.name]
        tsv_local_orig_op = self.ctx[self.TsvLocalOriginalOpId.name]
        tsv_ms_orig_op = self.ctx[self.TsvMsOriginalOpId.name]

        new_version = "{0}.r=bin_{1}.script_{2}".format(self.ctx[self.NewVersion.name], self.ctx[self.MatrinxetRevision.name], self.ctx[self.RunMatrinxetRevision.name])

        op_creator = OperationCreator(
            nirvana_api, token, self.ctx[self.StoragePath.name],
            mr_local_orig_op, mr_ms_orig_op, tsv_local_orig_op, tsv_ms_orig_op,
            new_version)

        op_creator.Create()

        # MR LOCAL
        with open('mr_local.json', 'wt') as resource_file:
            resource_file.write(nirvana_api.enc.encode({'op_id': op_creator.GetMrLocalOperation().GetNewId()}))

        self.create_resource("json with MR LOCAL Train matrixnet on CPU", 'mr_local.json', 'TRAIN_MATRIXNET_ON_CPU_MR_LOCAL_OP')

        # MR MS
        with open('mr_ms.json', 'wt') as resource_file:
            resource_file.write(nirvana_api.enc.encode({'op_id': op_creator.GetMrMsOperation().GetNewId()}))

        self.create_resource("json with MR MS Train matrixnet on CPU", 'mr_ms.json', 'TRAIN_MATRIXNET_ON_CPU_MR_MS_OP')

        # TSV LOCAL
        with open('tsv_local.json', 'wt') as resource_file:
            resource_file.write(nirvana_api.enc.encode({'op_id': op_creator.GetTsvLocalOperation().GetNewId()}))

        self.create_resource("json with TSV LOCAL Train matrixnet on CPU", 'tsv_local.json', 'TRAIN_MATRIXNET_ON_CPU_TSV_LOCAL_OP')

        # TSV MS
        with open('tsv_ms.json', 'wt') as resource_file:
            resource_file.write(nirvana_api.enc.encode({'op_id': op_creator.GetTsvMsOperation().GetNewId()}))

        self.create_resource("json with TSV MS Train matrixnet on CPU", 'tsv_ms.json', 'TRAIN_MATRIXNET_ON_CPU_TSV_MS_OP')
        st_token = self.get_vault_data(self.ctx[self.StTokenName.name])
        self.MakeStartrekCommentWithOperationsTable(
            st_token, self.ctx[self.StartrekTicket.name],
            op_creator.GetMrMsOperation().GetNewId(),
            op_creator.GetMrLocalOperation().GetNewId(),
            op_creator.GetTsvMsOperation().GetNewId(),
            op_creator.GetTsvLocalOperation().GetNewId(),
            self.ctx[self.MatrinxetRevision.name], self.ctx[self.RunMatrinxetRevision.name], new_version)


__Task__ = Task
