# -*- coding: utf-8 -*-
import os
import json
import time
import logging
import requests
import datetime

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.geosearch.tools.misc import retry
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.geosearch.startrek import StartrekClient


TIME_TO_SLEEP = 30 * 60
MAX_ERRORS_COUNT = 15


class MapsDatabaseBusinessQualityAcceptance(sdk2.Task):
    """
        Quality acceptance of maps database business
    """

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 6 * 60 * 60  # 6 hours
        mlm_template_path = sdk2.parameters.String('Arcadia path to MLM template',
                                                   default_value='arcadia:/arc/trunk/arcadia/search/metrics_templates/geo/base.json')
        startrek_task_id = sdk2.parameters.String('Id of startrek task',
                                                  required=True)
        vault_key_name = sdk2.parameters.String('Vault key name for MLM API',
                                                default_value='robot-thoth',
                                                required=True)
        mail_list = sdk2.parameters.List('List of emails to send report',
                                         default=['marvelstas@yandex-team.ru',
                                                  'abolkhovityanov@yandex-team.ru',
                                                  'evelkin@yandex-team.ru',
                                                  'karas-pv@yandex-team.ru',
                                                  'ninankara@yandex-team.ru',
                                                  'aleksko@yandex-team.ru',
                                                  'ppavel96@yandex-team.ru',
                                                  'asikorsky@yandex-team.ru'],
                                         required=True)
        prodution_hosts_list = sdk2.parameters.List('List of hosts on production',
                                                    default=['man1-5063.search.yandex.net:30100',
                                                             'man1-5064.search.yandex.net:30100',
                                                             'man1-7792.search.yandex.net:30100',
                                                             'man1-7793.search.yandex.net:30100'],
                                                    required=True)
        experimental_hosts_list = sdk2.parameters.List('List of hosts on production',
                                                       default=['sas1-7702.search.yandex.net:30100',
                                                                'sas1-7823.search.yandex.net:30100',
                                                                'sas1-7837.search.yandex.net:30100',
                                                                'sas1-7934.search.yandex.net:30100'],
                                                       required=True)
        production_sources = sdk2.parameters.List('"&search_experimental_source=" for reference stand')
        experimental_sources = sdk2.parameters.List('"&search_experimental_source=" for experimental stand')
        additional_cgi_params = sdk2.parameters.String('Additional CGI parameters',
                                                       default_value='')
        mark = sdk2.parameters.String('Launch mark')

    class Requirements(sdk2.Task.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass

    @sdk2.header()
    def head(self):
        if self.Context.mlm_task_id:
            return ('<a href="https://mlm.yandex-team.ru/sets/{mlm_id}">'
                    'MLM launch set'
                    '</a>').format(mlm_id=self.Context.mlm_task_id)
        else:
            return 'Processing...'

    def init_session(self):
        self.session = requests.Session()
        token = sdk2.Vault.data(self.Parameters.vault_key_name)
        self.session.headers['Authorization'] = 'OAuth {token}'.format(token=token)
        self.session.headers["Accept"] = "application/json"
        self.session.headers["Content-Type"] = "application/json"
        self.mlm_url = 'https://mlm.yandex-team.ru/api'
        self.startrek_client = StartrekClient(token)

    def _get_input_parameters(self):
        if self.Parameters.production_sources:
            self.reference_sources = ['search_experimental_source=%s&' % source for source in self.Parameters.production_sources]
        else:
            self.reference_sources = []
        if self.Parameters.experimental_sources:
            self.experimental_sources = ['search_experimental_source=%s&' % source for source in self.Parameters.experimental_sources]
        else:
            self.experimental_sources = []

    def _send_mail(self, mail_content, db_date):
        mail_to = 'marvelstas@yandex-team.ru'
        mail_cc = self.Parameters.mail_list
        subject = 'Quality acceptance of geosearch base'
        content_type = 'text/html'
        logging.info(mail_content)
        channel.sandbox.send_email(mail_to, mail_cc, subject, mail_content,
                                   content_type)

    def _get_launch_template(self):
        arcadia_path, template_name = os.path.split(self.Parameters.mlm_template_path)
        checkout_path = Arcadia.export(arcadia_path, './tempaltes')
        json_file = os.path.join(checkout_path, template_name)
        template = json.load(open(json_file))
        logging.info('Template type: %s' % type(template))
        return template

    def _patch_launch_templates(self):
        template = self._get_launch_template()
        launch_templates = template.get('launchTemplates')
        ref_src_string = ''.join(self.reference_sources)
        exp_src_string = ''.join(self.experimental_sources)
        for launch_template in launch_templates:
            new_servers = []
            for server in launch_template['servers']:
                cgi = server.get('cgi')
                merged_cgi = server.get('mergedCgi')
                new_server = server
                if server['name'] in ['production', 'Stable']:
                    new_server.update({'cgi': '{src}{cgi}'.format(src=ref_src_string, cgi=cgi),
                                       'mergedCgi': '{src}{merged_cgi}'.format(src=ref_src_string, merged_cgi=merged_cgi)})
                if server['name'] in ['experimental', 'Prestable']:
                    new_server.update({'cgi': '{src}{cgi}'.format(src=exp_src_string, cgi=cgi),
                                       'mergedCgi': '{src}{merged_cgi}'.format(src=exp_src_string, merged_cgi=merged_cgi)})
                new_servers.append(new_server)
            launch_template.update({'servers': new_servers})
        template.update({'launchTemplates': launch_templates})
        return template

    def _patch_mobile_launch_templates(self):
        template = self._get_launch_template()
        launch_templates = template.get('launchTemplates')
        for launch_template in launch_templates:
            new_servers = []
            for server in launch_template['servers']:
                cgi = '{mlm_cgi}&{additional_cgi}'.format(mlm_cgi=server.get('cgi'), additional_cgi=self.Parameters.additional_cgi_params)
                merged_cgi = '{mlm_cgi}&{additional_cgi}'.format(mlm_cgi=server.get('mergedCgi'), additional_cgi=self.Parameters.additional_cgi_params)
                new_server = server
                if server['name'] in ['production', 'Stable']:
                    new_server.update({'cgi': '%s&json_setup={"source": %s}' % (cgi, json.dumps(self.Parameters.production_sources)),
                                       'mergedCgi': '%s&json_setup={"source": %s}' % (merged_cgi, json.dumps(self.Parameters.production_sources))})
                if server['name'] in ['experimental', 'Prestable']:
                    new_server.update({'cgi': '%s&json_setup={"source": %s}' % (cgi, json.dumps(self.Parameters.experimental_sources)),
                                       'mergedCgi': '%s&json_setup={"source": %s}' % (merged_cgi, json.dumps(self.Parameters.experimental_sources))})
                new_servers.append(new_server)
            launch_template.update({'servers': new_servers})
        template.update({'launchTemplates': launch_templates})
        return template

    def patch_templates(self):
        if 'mobile_base.json' in self.Parameters.mlm_template_path:
            return self._patch_mobile_launch_templates()
        return self._patch_launch_templates()

    def _start_mlm_task(self, db_date):
        url = '%s/launch-set/' % self.mlm_url
        description = {'description': '%s [%s][%s]' % (self.Parameters.startrek_task_id,
                                                       self.Parameters.mark,
                                                       self.Context.db_date)}
        launch_templates = self.patch_templates()
        logging.info('Will use this launchTemplates: %s' % launch_templates)
        launch_templates.update(description)
        r = self.session.post(url, data=json.dumps(launch_templates))
        logging.info('URL: %s' % url)
        logging.info('Status code: %s' % r.status_code)
        logging.info('Response: %s' % r.text)
        response = r.json()
        mlm_task_id = response['id']
        logging.info(mlm_task_id)
        return mlm_task_id

    def _check_mlm_task_finished(self, mlm_task_id):
        url = '%s/launch-set/%s' % (self.mlm_url, mlm_task_id)
        errors_count = 0
        logging.info(self.Context.db_date)
        task_status = ''
        response = ''
        while True:
            try:
                r = self.session.get(url)
                response = r.json()
                logging.info(response)
                task_status = response['status']
                logging.info(task_status)
            except Exception as e:
                errors_count += 1
                if errors_count > MAX_ERRORS_COUNT:
                    raise SandboxTaskFailureError(
                        "Too many exceptions were catched! Last: " + str(e))
                time.sleep(60)
            if task_status == 'COMPLETED':
                return response
            elif task_status == 'FAILED' or task_status == 'CANCELED':
                raise SandboxTaskFailureError(
                    "MLM's task with id %s have status %s" %
                    (self.Context.mlm_task_id, task_status))
            else:
                raise sdk2.WaitTime(TIME_TO_SLEEP)

    def _get_color(self, signification):
        if signification is None or signification == 'NONE':
            return 'black'
        elif signification == 'RED':
            return 'red'
        elif signification == 'LIGHT_RED':
            return 'pink'
        elif signification == 'GRAY':
            return 'gray'
        elif signification == 'GREEN':
            return 'green'
        elif signification == 'LIGHT_GREEN':
            return 'chartreuse'
        else:
            logging.info('Unknown signification: ' + signification)
            return 'black'

    def _get_startrek_color(self, signification):
        if signification is None or signification == 'NONE':
            return '%s'
        elif signification == 'RED':
            return '**!!(red)%s!!**'
        elif signification == 'LIGHT_RED':
            return '!!(red)%s!!'
        elif signification == 'GRAY':
            return '!!(gray)%s!!'
        elif signification == 'GREEN':
            return '**!!(green)%s!!**'
        elif signification == 'LIGHT_GREEN':
            return '!!(green)%s!!'
        else:
            logging.info('Unknown signification: ' + signification)
            return '%s'

    @retry(tries=3, delay=2)
    def _get_docs_count_per_shard(self, host):
        url = 'http://' + host + '/?info=doccount'
        try:
            r = requests.get(url)
            response = r.text
            return int(response)
        except Exception as err:
            logging.info("Failed to get docs count from {url}".format(url=url))
            logging.info("Error {err}".format(err=err))
            return 1

    def _form_docs_count_report(self):
        production_docs_count = 0
        for host in self.Parameters.prodution_hosts_list:
            production_docs_count += self._get_docs_count_per_shard(host)

        experimental_docs_count = 0
        for host in self.Parameters.experimental_hosts_list:
            experimental_docs_count += self._get_docs_count_per_shard(host)

        max_docs_count = float(max(production_docs_count,
                               experimental_docs_count))
        docs_count_diff = float(abs(production_docs_count -
                                experimental_docs_count))
        docs_count_diff_percents = 100 * docs_count_diff / max_docs_count

        logging.info("Docs count in production base: " +
                     str(production_docs_count))
        logging.info("Docs count in experimental base: " +
                     str(experimental_docs_count))
        logging.info("Docs count diff: {0:.2f}%".format(
            docs_count_diff_percents))

        if docs_count_diff_percents > 10:
            color = 'red'
        else:
            color = 'black'

        report = 'Docs count in production base: ' +\
            str(production_docs_count) + '<br>'
        report += 'Docs count in experimental base: ' +\
            str(experimental_docs_count) + '<br>'
        report += 'Diff between docs count: <font color=' + color +\
            '>{0:.2f}%</font>'.format(docs_count_diff_percents) + '<br>'
        return report

    def _form_metrics_report(self, task_info):
        critical_metrics_count = 0
        warn_metrics_count = 0
        report = ''
        task_id = task_info['id']

        comments = []
        for launch in task_info['launches']:
            country = launch['name']
            launch_id = launch['id']
            launch_link = 'https://mlm.yandex-team.ru/sets/%s/%s' %\
                (task_id, launch_id)
            report += '<br><a href=%s>%s</a><br>' % (launch_link, country)
            report += '<table border=1><tr><th>Metric<th>Diff'
            comment = '**((%s %s %s))**\n' % (launch_link, country, '(%s)' % self.Parameters.mark)
            comment += '#|\n'
            comment += '|| **Metric** | **Diff** ||'

            for group in launch['diffQueryGroups']:
                comment += '|| **{0}** ||'.format(group['name'])
                for metric in group['metrics']:
                    metric_name = metric['metricName']
                    if type(metric['diffPercent']) == float:
                        metric_diff = "{0:.2f}%".format(metric['diffPercent'])
                    else:
                        metric_diff = '-'
                    metric_signification = metric['signification']
                    report += '<tr><td>' + metric_name
                    report += '<td><font color='
                    report += self._get_color(metric_signification)
                    report += '>' + metric_diff + '</font>'
                    comment += '|| %s | %s ||' %\
                        (metric_name,
                         self._get_startrek_color(metric_signification) %
                         metric_diff)
                    if country == 'Russia' and metric['status'] == 'CRITICAL':
                        critical_metrics_count += 1
                        logging.info('Metric %s is critical!' %
                                     metric['metricName'])
                    elif country == 'Russia' and metric['status'] == 'WARN':
                        warn_metrics_count += 1
                        logging.info('Metric %s have warning!' %
                                     metric['metricName'])
            report += '</table>'
            comment += '|#'
            comments.append('<{%s}>' % comment)
        if self.Parameters.startrek_task_id:
            self.startrek_client.add_comment(self.Parameters.startrek_task_id, '\n'.join(comments))

        metrics_summary = "Warning metric's count: " +\
            str(warn_metrics_count) + '<br>'
        metrics_summary += "Critical metric's count: " +\
            str(critical_metrics_count) + '<br>'
        report = metrics_summary + report
        logging.info(report)
        logging.info("Warning metric's count: %d" % warn_metrics_count)
        logging.info("Critical metric's count: %d" % critical_metrics_count)
        return report, critical_metrics_count

    def _accept_new_base(self, task_info, task_id, db_date):
        metrics_report, critical_metrics_count =\
            self._form_metrics_report(task_info)
        docs_count_report = self._form_docs_count_report()
        db_accepted = (critical_metrics_count == 0)
        mail_content = '<a href=https://mlm.yandex-team.ru/sets/' +\
            task_id + '>New geosearch base</a> was '

        if not db_accepted:
            mail_content += 'not '

        mail_content += 'accepted!<br>'
        mail_content += docs_count_report
        mail_content += metrics_report
        self._send_mail(mail_content, db_date)

        if critical_metrics_count > 0:
            raise SandboxTaskFailureError(
                "Critical metric's count > 0 !")

    def on_execute(self):
        self._get_input_parameters()
        self.init_session()

        if not self.Context.db_date:
            self.Context.db_date = datetime.datetime.now().strftime('%Y-%m-%dT%M:%H:%S')
        if self.Context.mlm_task_id:
            logging.info('Continuing previous MLM task: {mlm_task_id}'.format(mlm_task_id=self.Context.mlm_task_id))
        else:
            self.Context.mlm_task_id = self._start_mlm_task(self.Context.db_date)
            logging.info('Running new MLM task: %s' % self.Context.mlm_task_id)
            mlm_task_link = 'https://mlm.yandex-team.ru/sets/%s' % self.Context.mlm_task_id
            comment = 'New ((%s MLM task)) %s was started.' % (mlm_task_link, self.Parameters.mark)
            if self.Parameters.startrek_task_id:
                self.startrek_client.add_comment(self.Parameters.startrek_task_id, comment)
                self.startrek_client.add_followers(
                    self.Parameters.startrek_task_id,
                    [mail[:len(mail) - len('@yandex-team.ru')]
                        for mail in self.Parameters.mail_list])

        task_info = self._check_mlm_task_finished(self.Context.mlm_task_id)
        self._accept_new_base(task_info, self.Context.mlm_task_id, self.Context.db_date)
