# coding: utf-8

import logging
import random
import requests

from sandbox import sdk2

from sandbox.projects.common.search import instance_resolver as ir
from sandbox.projects.common.search import cluster_names
from sandbox.projects.common.nanny import client as nanny_client
from sandbox.projects.common import time_utils as tu
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine.security import get_rm_token
from sandbox.projects import resource_types
from sandbox.projects.websearch.middlesearch import resources as ms_resources

FROM_CLUSTERSTATE_CHOICE = "from_clusterstate"
SPECIFY_CHOICE = "specify"
FROM_NANNY_CHOICE = "from_nanny"


class MarkCurrentProdResources(sdk2.Task):
    """
        Mark current prod resources with special attribute and time
    """
    SEARCH_INFO = {
        "base": {
            "resource_types": [
                resource_types.DYNAMIC_MODELS_ARCHIVE_BASE,
                resource_types.BASESEARCH_EXECUTABLE,
            ],
            "c_conf": "vla-web-base-resources",
        },
        "middle": {
            "resource_types": [
                ms_resources.RankingMiddlesearchExecutable,
                ms_resources.IntsearchExecutable,
                resource_types.DYNAMIC_MODELS_ARCHIVE,
            ],
            "c_conf": cluster_names.C.vla_jupiter_mmeta,
        }
    }

    class Requirements(sdk2.Task.Requirements):
        ram = 1 * 1024
        disk_space = 2 * 1024  # 2 Gb
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String("Search type (base, middle, etc.)", required=True) as search_type:
            search_type.values.base = search_type.Value("base")
            search_type.values.middle = search_type.Value("middle")
        with sdk2.parameters.String("Instance source", required=True) as instance_source:
            instance_source.values.FROM_CLUSTERSTATE = instance_source.Value(FROM_CLUSTERSTATE_CHOICE)
            instance_source.values.SPECIFY = instance_source.Value(SPECIFY_CHOICE)
            instance_source.values.FROM_NANNY = instance_source.Value(FROM_NANNY_CHOICE, default=True)
        with instance_source.value.FROM_CLUSTERSTATE:
            clusterstate_conf = sdk2.parameters.String("Clusterstate configuration", required=False, default="")
            instance_tag = sdk2.parameters.String("Clusterstate instance tag", required=False, default="")
        with instance_source.value.SPECIFY:
            specified_instance = sdk2.parameters.String("Instance (ex. man1-1111:20000)")
        with instance_source.value.FROM_NANNY:
            pass

    def on_execute(self):
        if self.Parameters.instance_source == FROM_NANNY_CHOICE.upper():
            service_name = self.SEARCH_INFO[self.Parameters.search_type]["c_conf"].strip('-')
            active_attrs = self._get_nanny_service_runtime_attrs(service_name)
            resources = self._parse_resources_from_nanny_json(active_attrs)
        else:
            selected_instance = None
            logging.info("Instance source: %s", self.Parameters.instance_source)
            if self.Parameters.instance_source == FROM_CLUSTERSTATE_CHOICE.upper():
                c_conf = self.Parameters.clusterstate_conf or self.SEARCH_INFO[self.Parameters.search_type]["c_conf"]
                instances = ir.get_instances_by_config(c_conf)
                selected_instance = "{}:{}".format(*random.choice(instances))
            elif self.Parameters.instance_source == SPECIFY_CHOICE.upper():
                selected_instance = self.Parameters.specified_instance

            logging.info("Selected instance: %s", selected_instance)
            info_url = "http://{}/yandsearch?info=get_resources_info".format(selected_instance)
            resp = requests.get(info_url)
            logging.debug("Url: %s\nResponse: %s", resp.text, resp.request.url)
            resp.raise_for_status()

            resources = self._parse_get_resources_info(resp.json())

        curr_date = tu.date_ymdhm(sep="_")

        for resource in resources:
            self._mark_res(resource, curr_date)

    def _mark_res(self, res, curr_date):
        res.current_production = curr_date
        self.set_info("Mark resource: {} successful".format(res.id))

    def _get_nanny_service_runtime_attrs(self, service_name):
        nanny = nanny_client.NannyClient(rm_const.Urls.NANNY_BASE_URL, oauth_token=get_rm_token(self))
        return nanny.get_service_active_runtime_attrs(service_name)

    def _parse_resources_from_nanny_json(self, nanny_json):
        resources_set = set()

        lookup_resources_types_names = [
            r.name
            for r in self.SEARCH_INFO[self.Parameters.search_type]["resource_types"]
        ]
        sandbox_resources = nanny_json['content']['resources']['sandbox_files']

        for sb_resource in sandbox_resources:
            if sb_resource['resource_type'] in lookup_resources_types_names:
                resources_set.add(sdk2.Resource[sb_resource['resource_id']])

        return list(resources_set)

    def _parse_get_resources_info(self, resources_info_json):
        resources_set = set()

        lookup_resources_types = self.SEARCH_INFO[self.Parameters.search_type]["resource_types"]

        for item_key, item_id in resources_info_json.iteritems():
            if "resource" in item_key:
                resources_set.add(sdk2.Resource[item_id])
            if "task" in item_key:
                logging.info("Try to mark resource from task: {} successful".format(item_id))
                resources_set.update(
                    [
                        resource for resource in sdk2.Resource.find(task_id=item_id).limit(0)
                        if resource.type in lookup_resources_types
                    ]
                )

        return list(resources_set)
