# -*- coding: utf-8 -*-
import os.path

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

LXC_CONTAINER_ID = 464300862


class PdfResource(sdk2.Resource):
    document_type = sdk2.parameters.String("Document type")


class MarkdownToPdf(sdk2.Task):
    """
    Convert Markdown file to PDF.
    """

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        use_custom_markdown = sdk2.parameters.Bool("Use custom markdown")
        with use_custom_markdown.value[True]:
            input_markdown = sdk2.parameters.String("Input Markdown", multiline=True, required=True)
        with use_custom_markdown.value[False]:
            input_files = sdk2.parameters.List("Input Markdown file(s), relative to arcadia trunk", required=True)

        document_type = sdk2.parameters.String("Document type to be set as output resource 'document_type' attribute")

        css = sdk2.parameters.String(
            "CSS styles for intermediate .html, relative to arcadia trunk",
            default="sandbox/projects/MarkdownToPdf/github-pandoc.css",
            required=True
        )
        resource_dir = sdk2.parameters.String(
            "Optional resource directory (with images etc.) relative to arcadia trunk",
            description="Links to resources in markdown must be realtive and look like "
                        "[<basename resource_dir>/<resource_filename>]"
        )
        _container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=LXC_CONTAINER_ID,
            required=True
        )

    class Context(sdk2.Task.Context):
        pass

    def on_execute(self):
        """
        Convert Markdown to PDF using pandoc and electron-pdf and return data as resource.
        """
        def get_local_path(path):
            return os.path.abspath(os.path.basename(path))

        resources_path = get_local_path(self.Parameters.resource_dir)
        md_paths = [get_local_path("input.md")] if self.Parameters.input_markdown else [get_local_path(input_file) for input_file in self.Parameters.input_files]
        html_path = get_local_path("buffer.html")
        css_path = get_local_path(self.Parameters.css)
        pdf_path = get_local_path("result.pdf")

        pdf_resource_attrs = {"document_type": self.Parameters.document_type} if self.Parameters.document_type else {}
        pdf_resource = sdk2.ResourceData(PdfResource(self, "Output PDF", pdf_path, **pdf_resource_attrs))

        self.get_source_files(md_paths, css_path, resources_path)
        self.build_pdf(md_paths, css_path, html_path, pdf_path)
        pdf_resource.ready()

    def get_source_files(self, md_paths, css_path, resources_path):
        if self.Parameters.use_custom_markdown:
            with open(md_paths[0], "w") as f:
                f.write(self.Parameters.input_markdown.encode("utf-8"))
        else:
            for input_file, md_path in zip(self.Parameters.input_files, md_paths):
                sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url(input_file), md_path)

        sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url(self.Parameters.css), css_path)

        if self.Parameters.resource_dir:
            sdk2.svn.Arcadia.export(sdk2.svn.Arcadia.trunk_url(self.Parameters.resource_dir), resources_path)

    def build_pdf(self, md_paths, css_path, html_path, pdf_path):
        DISPLAY_ENV = ":99"
        env = {"DISPLAY": DISPLAY_ENV, "TMPDIR": "/tmp/"}

        with sdk2.helpers.ProcessLog(self, logger="pdf_generation") as pl:
            pl.logger.info("CWD: %s", os.getcwd())
            kwargs = dict(stdout=pl.stdout, stderr=sp.STDOUT, env=env)

            xvfb_pipe = sp.Popen(["Xvfb",  DISPLAY_ENV, "-screen", "0", "1920x1080x24"], **kwargs)

            sp.check_call(["pandoc"] + md_paths + ["-o", html_path, "--css", css_path, "-s"], **kwargs)
            sp.check_call(["electron-pdf", html_path, pdf_path, "-m", "0"], **kwargs)

            xvfb_pipe.terminate()
