# coding: utf-8

import os
import socket
import subprocess
import time

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter, ResourceSelector, SandboxBoolParameter


class CatalogerBin(ResourceSelector):
    name = 'cataloger_bin'
    description = 'Cataloger binary. Empty to build from trunk'
    resource_type = resource_types.MARKET_CATALOGER_BIN
    required = False


class CatalogerData(ResourceSelector):
    name = 'cataloger_data'
    description = 'Cataloger data. Empty to auto-find'
    resource_type = resource_types.MARKET_CATALOGER_DATA
    required = False


class CatalogerRequests(ResourceSelector):
    name = 'cataloger_requests'
    description = 'Cataloger requests'
    resource_type = resource_types.PLAIN_TEXT_QUERIES
    default_value = 184057567  # grepped from logs
    required = True


class GraphiteHost(SandboxStringParameter):
    name = 'graphite_host'
    description = 'Graphite host to push data. Empty to ignore'
    default_value = 'graphite-testing.market.yandex.net'


class DebugMode(SandboxBoolParameter):
    name = 'debug_mode'
    description = 'Debug mode'


class MarketCatalogerShooter(SandboxTask):
    """
        Таск приемки каталогера на производительность
    """
    type = 'MARKET_CATALOGER_SHOOTER'
    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.INTEL_E5_2650

    input_parameters = (
        CatalogerBin,
        CatalogerData,
        CatalogerRequests,
        GraphiteHost,
        DebugMode
    )

    binary_type = resource_types.MARKET_CATALOGER_BIN
    binary_ctx_name = CatalogerBin.name
    data_type = resource_types.MARKET_CATALOGER_DATA
    data_ctx_name = CatalogerData.name
    binary_build_task_type = 'BUILD_MARKET_CATALOGER_BIN'
    daemon_name = 'cataloger'
    reqs_ctx_name = CatalogerRequests.name
    reqs_limit = 100000
    time_to_start = 120
    total_sessions = 5

    @staticmethod
    def generate_config(data_path):
        tpl = """CATALOG_FILENAME                 {data}/catalog.xml
COOKBOOKS_FILENAME               {data}/guru.cookbooks.xml
FIXED_TIME                       20151005_1511
GLOBAL_VENDORS_FILENAME          {data}/global.vendors.xml
GL_REGION_STATS_FILENAME         {data}/guru_light_region_stats.csv
LOG                              marketcataloger.log
MOBILE_CATALOG_FILENAME          {data}/mobile_catalog.xml
NAVIGATION_INFO                  {data}/navigation_info.xml
OFFERS_FILENAME                  {data}/data.xml
REGION_STATS_FILENAME            {data}/category_region_stats.csv
REGION_TREE_FILENAME             {data}/geo.c2p
SHOPSTAT_FILENAME                {data}/shopstat.xml
SHOW_LOG_PATH                    marketcataloger_shows.log
TOP_CATEGORIES_FILENAME          {data}/top_categories.xml
VENDORS_GURU_CATEGORIES_FILENAME {data}/vendors_guru_categories.csv"""
        return tpl.format(data=data_path)

    def push_rps(self, rps):
        if not self.ctx.get(GraphiteHost.name):
            return
        s = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        s.connect((self.ctx[GraphiteHost.name], 2024))
        s.sendall('one_hour.%s.shootings.rps %s %s' % (self.daemon_name, rps, int(time.time())))
        s.close()

    def build_binary(self):
        ctx_name = 'binary_build_task'
        if ctx_name not in self.ctx:
            self.ctx[ctx_name] = self.create_subtask(
                task_type=self.binary_build_task_type,
                description='build %s from trunk' % self.daemon_name
            ).id
            self.wait_task_completed(self.ctx[ctx_name])
        else:
            self.ctx[self.binary_ctx_name] = apihelpers.get_task_resource_id(
                task_id=self.ctx[ctx_name],
                resource_type=self.binary_type,
            )

    def on_execute(self):
        if self.ctx[DebugMode.name]:
            self.total_sessions = 1
            self.reqs_limit = 1000

        if not self.ctx.get(self.binary_ctx_name):
            self.build_binary()

        bin_id = self.ctx[self.binary_ctx_name]
        data_id = self.ctx[self.data_ctx_name] if self.ctx[self.data_ctx_name] else apihelpers.get_last_released_resource(self.data_type).id
        reqs_id = self.ctx[self.reqs_ctx_name]

        data_path = self.sync_resource(data_id)
        bin_path = self.sync_resource(bin_id)
        reqs_path = self.sync_resource(reqs_id)

        config_path = '%s.cfg' % self.daemon_name
        with open(config_path, 'w') as config:
            config.write(self.generate_config(data_path))

        reqs = []
        for n, line in enumerate(open(reqs_path)):
            reqs.append(line.replace('\n', ''))
            if n > self.reqs_limit:
                break

        sum_rps = 0
        for session in range(self.total_sessions):
            process = run_process(
                cmd='{} -c {}'.format(bin_path, config_path).split(),
                log_prefix=self.daemon_name,
                stdin=subprocess.PIPE,
                stdout=open(os.devnull, 'w') if not self.ctx[DebugMode.name] else None,
                wait=False
            )
            time.sleep(self.time_to_start)  # to let daemon start
            self.ctx['time_start_%s' % session] = time.time()
            process.communicate('\n'.join(reqs))
            self.ctx['time_finish_%s' % session] = time.time()
            self.ctx['time_delta_%s' % session] = self.ctx['time_finish_%s' % session] - self.ctx['time_start_%s' % session]
            rps = len(reqs) / self.ctx['time_delta_%s' % session]
            sum_rps += rps
            self.ctx['rps_%s' % session] = rps

        self.ctx['rps'] = int(sum_rps / self.total_sessions)
        self.push_rps(self.ctx['rps'])


__Task__ = MarketCatalogerShooter
