import os
import logging

import sandbox.sdk2 as sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.sandboxsdk.environments as environments
import sandbox.common.types.resource as ctr
import sandbox.common.errors as errors

from sandbox.projects.resource_types import MR_GEMINICL

from sandbox.projects.MarketModelWizard.PrepareOfferTables.yql_queries import first_part_query

yt_host_to_market_indexer = {
    'hahn': 'stratocaster',
    'banach': 'gibson',
}


class MarketModelWizardPrepareOfferYtExportExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['zhshishkin']


class MarketModelWizardPrepareOfferYtExport(sdk2.Task):
    """Prepare export fot ferryman"""

    class Parameters(sdk2.Task.Parameters):
        resource_id = sdk2.parameters.LastReleasedResource(
            'Executable resource Id',
            resource_type=MarketModelWizardPrepareOfferYtExportExecutable,
            state=(ctr.State.READY,),
            required=True,
        )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            common_folder = sdk2.parameters.String(
                'Common folder',
                required=True,
            )

            path = sdk2.parameters.String(
                'Result output table',
                required=True,
            )

            yt_vault_token = sdk2.parameters.String(
                'Your yt token name in vault',
                default='yt-token',
                required=True
            )

            yql_vault_token = sdk2.parameters.String(
                'Your yql token name in vault',
                default='yql-token',
                required=True
            )

            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(value='Hahn')
                yt_host.values['banach'] = yt_host.Value(value='Banach', default=True)

        with sdk2.parameters.Group('Gemini parameters') as gemini_parameters:
            gemini_resource_id = sdk2.parameters.Resource(
                'MR Geminicl',
                resource_type=MR_GEMINICL,
                state=(ctr.State.READY,),
                required=True,
            )

            mr_gemini_user = sdk2.parameters.String(
                'MR gemini user',
                default='mr-any',
                required=True,
            )
            mr_gemini_job_count = sdk2.parameters.Integer(
                'MR gemini job count',
                default=1000,
                required=True,
            )
            mr_gemini_max_rps = sdk2.parameters.Integer(
                'MR gemini max rps',
                default=50000,
                required=True,
            )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql', version='1.2.49'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt
        from yql.api.v1.client import YqlClient

        yql_client = YqlClient(
            db=self.Parameters.yt_host,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_vault_token),
        )

        market_indexer = yt_host_to_market_indexer[self.Parameters.yt_host]

        yt_proxy = '{}.yt.yandex.net'.format(self.Parameters.yt_host)

        yt.config['token'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
        yt.config['proxy']['url'] = yt_proxy

        gemini_runner = self.Parameters.gemini_resource_id

        if gemini_runner is None:
            raise errors.TaskError('No mr_geminicl founded')
        gemini_runner = sdk2.ResourceData(gemini_runner)

        common_path = str(self.Parameters.common_folder)

        url_table = os.path.join(common_path, 'raw_urls')
        canonized_url_table = os.path.join(common_path, 'canonized_urls')
        gemini_table = os.path.join(common_path, 'gemini_urls')
        gemini_doc_table = os.path.join(common_path, 'gemini_docids')

        with self.memoize_stage.yql_query(commit_on_entrance=False):
            request = yql_client.query(
                first_part_query.format(
                    market_indexer=market_indexer,
                    result_table=url_table,
                )
            )
            request.run()
            request.get_results(wait=True)

            if not request.is_success:
                raise errors.TaskError('YQL query failed')

        logging.info('Start first gemini run')
        with sdk2.helpers.ProcessLog(self, logger='first mr_gemini') as pl:
            env = os.environ.copy()
            env['MR_RUNTIME'] = 'YT'
            env['MR_TMP'] = '//tmp/'
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)

            run = [
                str(gemini_runner.path),
                '--server', yt_proxy,
                '--user', str(self.Parameters.mr_gemini_user),
                '--input', str(url_table),
                '--output', str(gemini_table),
                '--error-output', yt.create_temp_table(),
                '--job-count', str(self.Parameters.mr_gemini_job_count),
                '--max-rps', str(self.Parameters.mr_gemini_max_rps),
                '--type', 'weak',
                '--substitute-keys',
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('first gemini run is failed')

        runner = self.Parameters.resource_id

        if runner is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(runner)

        with sdk2.helpers.ProcessLog(self, logger='first yt_runner') as pl:
            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            env['YT_PROXY'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)
            run = [
                str(runner.path),
                '--input-table', gemini_table,
                '--output-table', canonized_url_table,
                '--stage', 'first',
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('run is failed')

        with sdk2.helpers.ProcessLog(self, logger='second mr_gemini') as pl:
            env = os.environ.copy()
            env['MR_RUNTIME'] = 'YT'
            env['MR_TMP'] = '//tmp/'
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)

            run = [
                str(gemini_runner.path),
                '--server', yt_proxy,
                '--user', str(self.Parameters.mr_gemini_user),
                '--input', str(canonized_url_table),
                '--output', str(gemini_doc_table),
                '--error-output', yt.create_temp_table(),
                '--job-count', str(self.Parameters.mr_gemini_job_count),
                '--max-rps', str(self.Parameters.mr_gemini_max_rps),
                '--type', 'search_doc_id',
                '--substitute-keys',
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('second gemini run is failed')

        runner = self.Parameters.resource_id

        if runner is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(runner)

        with sdk2.helpers.ProcessLog(self, logger='second yt_runner') as pl:
            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            env['YT_PROXY'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)
            run = [
                str(runner.path),
                '--input-table', gemini_doc_table,
                '--output-table', str(self.Parameters.path),
                '--stage', 'second',
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('run is failed')
