import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common import file_utils as fu
from sandbox.common import errors
import sandbox.common.types.resource as ctr
import sandbox.sandboxsdk.environments as sdk_environments

from sandbox.projects.MarketModelWizard import GetDocidsFromXmlSearchResultTsv


def iter_over_docid(lines):
    for line in lines:
        splitted = line.split('\t')
        model_id, geo_id, _ = splitted[:3]
        for docid in splitted[3:]:
            if docid:
                yield {
                    'docid': docid,
                    'model_id': model_id,
                    'geo_id': geo_id,
                }


class MarketModelWizardPrepareSerpYtExportExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['zhshishkin']


class MarketModelWizardPrepareSerpYtExport(sdk2.Task):
    """Prepare export fot ferryman"""

    class Parameters(sdk2.Task.Parameters):
        resource_id = sdk2.parameters.LastReleasedResource(
            'Executable resource Id',
            resource_type=MarketModelWizardPrepareSerpYtExportExecutable,
            state=(ctr.State.READY,),
            required=True,
        )

        input_tsv = sdk2.parameters.Resource(
            'Input tsv table (id, geoid, text, docids)',
            resource_type=GetDocidsFromXmlSearchResultTsv,
        )

        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            common_folder = sdk2.parameters.String(
                'Common folder',
                required=True,
            )
            path = sdk2.parameters.String(
                'Result output table',
                required=True,
            )

            yt_vault_token = sdk2.parameters.String(
                'Your token name in vault',
                default='yt-token',
                required=True)

            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(value='Hahn')
                yt_host.values['banach'] = yt_host.Value(value='Banach', default=True)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment('yandex-yt'),
            sdk_environments.PipEnvironment('yandex-yt-yson-bindings'),
            sdk_environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

    def on_execute(self):
        input_table = input_table = os.path.join(self.Parameters.common_folder, 'input')

        with self.memoize_stage.upload_data:
            import yt.wrapper as yt

            yt.config['token'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)

            input_tsv_data = sdk2.ResourceData(self.Parameters.input_tsv)

            yt.write_table(
                input_table,
                iter_over_docid(fu.read_line_by_line(str(input_tsv_data.path))),
            )

        runner = self.Parameters.resource_id

        if runner is None:
            raise errors.TaskError('No executable founded')
        runner = sdk2.ResourceData(runner)

        with sdk2.helpers.ProcessLog(self, logger='yt_runner') as pl:

            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            env['YT_PROXY'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)
            run = [
                str(runner.path),
                '--input-table', str(input_table),
                '--common-folder', str(self.Parameters.common_folder),
                '--output-table', str(self.Parameters.path)
            ]
            ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
            if ret:
                raise errors.TaskError('run is failed')
