# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
import requests

from collections import namedtuple
from datetime import datetime

from sandbox import sdk2
from sandbox.sandboxsdk import environments

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
WIKI_API = 'https://wiki-api.yandex-team.ru/_api/frontend/{}/.grid'

SQL = """
USE hahn;
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

SELECT *
FROM `//home/antifraud/export/market/shops/daily/clones2/latest`
WHERE clone_id in [{shop_ids}] or shop_id in [{shop_ids}]
"""


def execute_query(query, token):
    from yql.api.v1.client import YqlClient
    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        pattern = namedtuple('Record', result.column_names)
        for record in result.get_iterator():
            yield pattern(*record)


def collect_previous_issues(token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    filter_ = {'queue': 'BQ', 'components': ['112723'],
               'created': {'from': str(datetime.now().date())}}

    previous_shop_ids = []
    for issue in client.issues.find(filter=filter_, perPage=100):
        previous_shop_ids.append(issue.idMagazina)

    return previous_shop_ids


class ClonesAntifraud(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = ''
        max_restarts = 2
        kill_timeout = 20 * 60
        fail_on_any_error = True
        dump_disk_usage = False

        st_token = sdk2.parameters.YavSecret('OAuth ST token', required=True)
        yt_token = sdk2.parameters.YavSecret('OAuth YQL token', required=True)

    @staticmethod
    def create_issue(host, token):
        from startrek_client import Startrek
        client = Startrek(useragent='market-quality-control',
                          base_url=STARTREK_URL, token=token)

        summary = 'Антифрод нашел потенциального клона ID: {}'.format(host.clone_id)
        description = 'ID магазинов, с которым было найдено пересечение: {}\n'.format(host.shop_id)\
            + 'Тип магазина в пересечении: {}\n'.format(host.shop_type)\
            + 'Домен магазина в пересечении: {}'.format(host.shop_domain)

        client.issues.create(
            queue='BQ',
            summary=summary,
            type={'name': 'Задача'},
            description=description,
            components=[112723],
            idMagazina=host.clone_id
        )

    def on_execute(self):
        st_token = self.Parameters.st_token.data()
        st_token = st_token[self.Parameters.st_token.default_key]
        yt_token = self.Parameters.yt_token.data()
        yt_token = yt_token[self.Parameters.yt_token.default_key]

        ids = []
        response = requests.get(
            url=WIKI_API.format('users/hitinap/magazinyvykljuchennyezamoshennichestvo'),
            headers={
                'Authorization': 'OAuth {}'.format(st_token),
                'Content-type': 'application/json',
                'Accept-Charset': 'UTF-8'
            },
            timeout=1200
        )

        json_data = response.json()
        data_rows = json_data['data']['rows']
        for row in data_rows:
            try:
                ids.append(int(row[0]['raw']))
            except:
                continue

        previous_shop_ids = collect_previous_issues(st_token)
        for record in execute_query(SQL.format(shop_ids=','.join(map(str, ids))), token=yt_token):
            if record.clone_id not in previous_shop_ids:
                self.create_issue(record, token=st_token)
