# -*- coding: utf-8 -*-
import os
import requests

from collections import namedtuple
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments

CUTOFFS = {
    1000: 'Отключение СКК по причине качества',
    1001: 'Нарушены предусловия для включения фичи',
    1002: 'Отключено менеджером СКК',
    1003: 'Отключение по просьбе партнера',
    1004: 'На тестировании (Самопроверка)',
    1005: 'BALANCE_SUSPENDED',
    1006: 'Отключение программы для проведения эксперимента',
    1007: 'Отключение пингером',
    1008: 'Установлен лимит на количество выполненных заказов',
    1009: 'Ручное отключение программы',
    1010: 'Отключение программы со скрытием',
    1012: 'Отключение за низкий рейтинг качества',
    1013: 'Отключение за массовый картдифф',
    1014: 'Отключение за пропущенные заказы',
    1015: 'Отключение по просьбе партнера',
    1016: 'Тестинг',
    1019: 'Скрытие за несобранный в срок заказ',
    1020: 'Скрытие за клоновость',
    # DBS
    107: 'Отключение программы за непринятый вовремя заказ',
    108: 'Отключение программы за неприемлемое качество',
    109: 'Отключение программы за прочие проблемы качества',
    110: 'Отключение программы по превышению дневного лимита',
    111: 'Отключение из-за приостановки модерации',
    112: 'Отключение за невыставленные финальные статусы заказов',
    113: 'Не прошла проверка качества',
    # Common
    100: 'FEED',
    101: 'Мошенничество',
    102: 'Отключение за клоновость',
    103: 'Отключение за массовый картдифф'
}


class DailyCutoffNotifier(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('pandas'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('psycopg2-binary'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Telegram notifier'
        max_restarts = 2
        kill_timeout = 10 * 60
        fail_on_any_error = True
        dump_disk_usage = False

        yt_token = sdk2.parameters.YavSecret('OAuth YT token', required=True)
        tg_token = sdk2.parameters.YavSecret('Telegram bot token', required=True)

        with sdk2.parameters.Group('Database parameters') as database_parameters:
            host = sdk2.parameters.String('Host', default='c-mdbj7t5km7hpm8ugee52.rw.db.yandex.net', required=True)
            port = sdk2.parameters.Integer('Port', default=6432, required=True)
            dbname = sdk2.parameters.String('Database', required=True)
            user = sdk2.parameters.String('User', required=True)
            password = sdk2.parameters.YavSecret('Password', required=True)
            sslmode = sdk2.parameters.String('sslmode', default='require', required=True)

        with sdk2.parameters.Group('ABO db parameters') as database_parameters:
            abo_host = sdk2.parameters.String('Host', default='abo-market01f.db.yandex.net', required=True)
            abo_port = sdk2.parameters.Integer('Port', default=6432, required=True)
            abo_dbname = sdk2.parameters.String('Database', required=True)
            abo_user = sdk2.parameters.String('User', required=True)
            abo_password = sdk2.parameters.YavSecret('Password', required=True)
            abo_sslmode = sdk2.parameters.String('sslmode', default='require', required=True)

    @staticmethod
    def execute_yql(query, token=None):
        from yql.api.v1.client import YqlClient
        client = YqlClient(token=token)
        request = client.query(query)
        request.run()

        for result in request.get_results():
            pattern = namedtuple('Record', result.column_names)
            for record in result.get_iterator():
                yield pattern(*record)

    @staticmethod
    def send_message(cutoffs_data, today, token):
        message = '**{}**\n\n'.format(today)
        parameters = {}
        for row in cutoffs_data:
            parameters.setdefault(row['program'], {'enable': [], 'disable': []})
            if row['status'] == 'Включен':
                parameters[row['program']]['enable'].append(row)
            else:
                parameters[row['program']]['disable'].append(row)

        message += '\n'.join([
        '`{model}`: Открыто отключений - {all} ({disable}/{enable})'.format(
            model=p,
            all=len(parameters[p]['enable'] + parameters[p]['disable']),
            disable=len(parameters[p]['disable']),
            enable=len(parameters[p]['enable']),
        ) for p in parameters])
        message += '\n\n(Не размещаются/Включились)'

        method = 'https://api.telegram.org/bot{}/sendMessage'.format(token)
        channels = ['-1001687799501', '-1001213579991']
        for channel_id in channels:
            requests.post(
                url=method,
                data={
                    'chat_id': channel_id,
                    'text': message,
                    'parse_mode': 'Markdown',
                },
            )
            requests.post(
                url='https://api.telegram.org/bot{}/sendDocument'.format(token),
                data={'chat_id': channel_id},
                files={
                    'document': open(today + '.csv', 'rb')
                },
            )

    def on_execute(self):
        from psycopg2.extras import NamedTupleCursor
        from psycopg2 import connect
        from pandas import DataFrame

        yt_token = self.Parameters.yt_token.data()
        yt_token = yt_token[self.Parameters.yt_token.default_key]
        tg_token = self.Parameters.tg_token.data()
        tg_token = tg_token[self.Parameters.tg_token.default_key]

        db_config = {
            'host': self.Parameters.host,
            'port': self.Parameters.port,
            'dbname': self.Parameters.dbname,
            'user': self.Parameters.user,
            'password': self.Parameters.password.data()[self.Parameters.password.default_key],
            'sslmode': self.Parameters.sslmode,
        }
        abo_db_config = {
            'host': self.Parameters.abo_host,
            'port': self.Parameters.abo_port,
            'dbname': self.Parameters.abo_dbname,
            'user': self.Parameters.abo_user,
            'password': self.Parameters.abo_password.data()[self.Parameters.abo_password.default_key],
            'sslmode': self.Parameters.abo_sslmode,
        }

        today = str(datetime.now().date())
        yesterday = str((datetime.now() - timedelta(days=1)).date())

        daily_cutoffs = {}

        with connect(cursor_factory=NamedTupleCursor, **db_config).cursor() as cursor:
            with open(os.path.join(os.path.dirname(__file__), 'dbs_open_cutoff.sql'), 'r') as f:
                query = f.read()

            cursor.execute(query.format(yesterday, today))
            records = cursor.fetchall()
            daily_cutoffs = {row.datasource_id: row._asdict() for row in records}

            with open(os.path.join(os.path.dirname(__file__), 'supplier_open_cutoff.sql'), 'r') as f:
                query = f.read()

            cursor.execute(query.format(yesterday, today))
            records = cursor.fetchall()
            daily_cutoffs.update({row.datasource_id: row._asdict() for row in records})

            with open(os.path.join(os.path.dirname(__file__), 'dbs_closed_cutoff.sql'), 'r') as f:
                query = f.read()

            cursor.execute(query.format(yesterday, today))
            records = cursor.fetchall()
            daily_cutoffs.update({row.datasource_id: row._asdict() for row in records})

            with open(os.path.join(os.path.dirname(__file__), 'supplier_closed_cutoff.sql'), 'r') as f:
                query = f.read()

            cursor.execute(query.format(yesterday, today))
            records = cursor.fetchall()
            daily_cutoffs.update({row.datasource_id: row._asdict() for row in records})

        with connect(cursor_factory=NamedTupleCursor, **abo_db_config).cursor() as cursor:
            cursor.execute("""SELECT DISTINCT shop_id
            FROM recheck_ticket
            WHERE modification_time > '{}'
                AND modification_time < '{}'
                AND status_id NOT IN (1, 4, 5)
                AND ya_uid <> 0
                AND type_id = 15
            """.format(yesterday, today))
            records = cursor.fetchall()

        daily_cutoffs = daily_cutoffs.values()
        recheck_tickets = [row.shop_id for row in records]
        cutoffs_data = []

        for row in daily_cutoffs:
            if row['datasource_id'] not in recheck_tickets:
                row['status'] = 'Включен' if row['status'] == 'SUCCESS' else 'Не размещается'
                row['cutoff_name'] = CUTOFFS.get(row['cutoff_type'], 'UNKNOWN')
                cutoffs_data.append(row)

        yql_query = """\
        USE hahn;
        $state_b2b_partner = (
            SELECT MAX(Path) FROM FOLDER("//home/market/production/mstat/dwh/calculation/state_b2b_partner")
        );

        SELECT supplier_id
            , supplier_last_30_days_created_gmv_rub_numeric
            , manager_kd_name
            , manager_kd_ld_login
        FROM CONCAT($state_b2b_partner)
        WHERE supplier_id IN ({})
        """.format(','.join(str(row['datasource_id']) for row in cutoffs_data))

        query_results = {}
        for row in self.execute_yql(yql_query, token=yt_token):
            query_results[row.supplier_id] = {
                'gmv': row.supplier_last_30_days_created_gmv_rub_numeric,
                'manager_kd_name': row.manager_kd_name,
                'manager_kd_ld_login': row.manager_kd_ld_login,
            }

        to_send_data = []
        for row in cutoffs_data:
            if row['datasource_id'] in query_results:
                row['gmv'] = query_results[row['datasource_id']]['gmv']
                row['manager_name'] = query_results[row['datasource_id']]['manager_kd_name']
                row['manager_login'] = query_results[row['datasource_id']]['manager_kd_ld_login']
            to_send_data.append(row)

        df = DataFrame(to_send_data)
        df.to_csv(os.getcwd() + '/' + today + '.csv', index=False)
        self.send_message(to_send_data, today, tg_token)
