# -*- coding: utf-8 -*-
import requests
import xml.etree.ElementTree as ET

from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def get_weekends():
    today = datetime.now().date()
    month = (datetime.now() - timedelta(days=30)).date()

    response = requests.get(
        url='https://api.calendar.yandex-team.ru/export/holidays.xml?'
            'start_date={}&end_date={}&country_id=225&out_mode=all'.format(month, today)
    )

    root = ET.fromstring(response.text)
    holidays = list(root.find('get-holidays').find('days').iter('day'))

    for day in holidays:
        if int(day.attrib['is-holiday']) == 1:
            yield day.attrib['date']


def calculate_time_range(creation_time, modification_time, weekends):
    days_range = (modification_time - creation_time).days
    distance = [creation_time]
    for i in range(1, days_range):
        day = (creation_time + timedelta(days=i)).date()
        distance.append(datetime.strptime(day, '%Y-%m-%d'))
    distance.append(modification_time)

    time_range = timedelta()
    for i in range(len(distance)-1):
        current_date = distance[i].date()
        next_date = distance[i+1].date()

        if str(next_date) in weekends and str(current_date) in weekends:
            continue
        elif str(current_date) in weekends:
            time_range += distance[i+1] - datetime.strptime(str(next_date), '%Y-%m-%d')
        elif str(next_date) in weekends:
            time_range += datetime.strptime(str(next_date), '%Y-%m-%d') - distance[i]
        else:
            time_range += distance[i+1] - distance[i]

    return time_range


MESSAGE = 'Медиана времени проверки DBS > 8 часов: '\
    '[DataLens](https://datalens.yandex-team.ru/tewnnsjl6ou4m-sla?state=73b2c19c221)'


class DbsTimeAlarm(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('psycopg2-binary'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Telegram notifier'
        max_restarts = 2
        kill_timeout = 10 * 60
        fail_on_any_error = True
        dump_disk_usage = False

        st_token = sdk2.parameters.YavSecret('OAuth ST token', required=True)
        tg_token = sdk2.parameters.YavSecret('Telegram bot token', required=True)

        with sdk2.parameters.Group('Database parameters') as database_parameters:
            host = sdk2.parameters.String('Host', default='abo-market01f.db.yandex.net', required=True)
            port = sdk2.parameters.Integer('Port', default=6432, required=True)
            dbname = sdk2.parameters.String('Database', required=True)
            user = sdk2.parameters.String('User', required=True)
            password = sdk2.parameters.YavSecret('Password', required=True)
            sslmode = sdk2.parameters.String('sslmode', default='require', required=True)

    @staticmethod
    def find_issues_by_today(token):
        from startrek_client import Startrek
        client = Startrek(useragent='market-qc', base_url=STARTREK_URL, token=token)

        today = str(datetime.now().date())
        filter_ = {'queue': 'TURBOTEST', 'components': ['116449'], 'created': {'from': today}}

        for issue in client.issues.find(filter=filter_, perPage=100):
            yield issue.key

    @staticmethod
    def create_issue(token):
        from startrek_client import Startrek
        client = Startrek(useragent='market-qc',
                          base_url=STARTREK_URL, token=token)

        client.issues.create(
            queue='TURBOTEST',
            summary='Медиана времени проверки DBS > 8 часов',
            type={'name': 'Задача'},
            description='',
            components=[116449]
        )

    def send_message(self, st_token, tg_token):
        method = 'https://api.telegram.org/bot{}/sendMessage'.format(tg_token)
        requests.post(
            method, data={
                "chat_id": '885388542',
                "text": MESSAGE,
                "parse_mode": "Markdown"
            }
        )
        self.create_issue(st_token)

    def on_execute(self):
        from psycopg2.extras import NamedTupleCursor
        from psycopg2 import connect
        import pytz

        st_token = self.Parameters.st_token.data()
        st_token = st_token[self.Parameters.st_token.default_key]
        tg_token = self.Parameters.tg_token.data()
        tg_token = tg_token[self.Parameters.tg_token.default_key]

        issues_today = list(self.find_issues_by_today(st_token))
        if len(issues_today) > 0:
            return

        db_config = {
            'host': self.Parameters.host,
            'port': self.Parameters.port,
            'dbname': self.Parameters.dbname,
            'user': self.Parameters.user,
            'password': self.Parameters.password.data()[self.Parameters.password.default_key],
            'sslmode': self.Parameters.sslmode,
        }

        with connect(cursor_factory=NamedTupleCursor, **db_config).cursor() as cursor:
            cursor.execute("""SELECT id ticket_id
                , creation_time
                , modification_time
                , assessor_holds
                , status_id
            FROM v_yt_exp_premod_ticket
            WHERE creation_time > current_date at time zone 'Europe/Moscow'
                AND check_type = 3
                AND status_id <> 5
            """)
            records = cursor.fetchall()

        time_ranges = []
        weekends = list(get_weekends())
        time_format = '%Y-%m-%dT%H:%M:%S.%f+03:00'

        for row in records:
            modification_time = row.modification_time
            if row.status_id not in [3, 4, 5, 8, 9]:
                modification_time = datetime.now().replace(tzinfo=pytz.timezone('Europe/Moscow'))

            ticket_time = calculate_time_range(row.creation_time, modification_time, weekends)
            total_hold_time = timedelta()
            for obj in row.assessor_holds['assessor_holds']:
                start_time = datetime.strptime(obj['creation_time'], time_format)
                release_time = datetime.strptime(obj['plan_release_time'], time_format)
                total_hold_time += calculate_time_range(start_time, release_time, weekends)

            ticket_time = (ticket_time - total_hold_time).total_seconds()
            if ticket_time < 0:
                ticket_time = 0

            time_ranges.append(ticket_time)

        def median(array):
            index = len(array) // 2
            if len(array) % 2:
                return sorted(array)[index]
            return sum(sorted(array)[index - 1:index + 1]) / 2

        median_time = median(time_ranges)
        if median_time > 8:
            self.send_message(st_token, tg_token)
