# -*- coding: utf-8 -*-

import logging
from json import dumps
from random import shuffle, choice
from requests import get, post, patch
from datetime import timedelta, datetime, time

from sandbox import sdk2
from sandbox.sandboxsdk import environments

GRID = 'users/hitinap/deliveryexceptions'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
DEPARTMENT_EXCEPTIONS = ['Бригада регулярных авто-проверок (Поддержка бизнеса)',
                         'Бригада резервных асессоров (Поддержка бизнеса)']


class Assignees_selector:
    assignees = []
    users_workload = {}

    @staticmethod
    def _delete_expired_records(token, version, expired_rows):
        url = 'https://wiki-api.yandex-team.ru/_api/frontend/%s/.grid/change' % GRID
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        data = {
            'version': version,
            'changes': [
                {
                    'removed_row': {
                        'id': id
                    }
                }
                for id in expired_rows]
        }
        post(url, data=dumps(data, ensure_ascii=False).encode('UTF-8'), headers=headers)

    def _get_exception_list(self, token):
        url = 'https://wiki-api.yandex-team.ru/_api/frontend/%s/.grid' % GRID
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        req = get(url, headers=headers).json()

        table = req['data']['rows']
        version = req['data']['version']
        logins = [row[0]['raw'][0] for row in table if len(row[0]['raw']) != 0]
        expired_rows = [row[2]['row_id'] for row in table if row[2]['raw'] != ''
                        and (datetime.now() - datetime.strptime(row[2]['raw'], '%Y-%m-%d')).days > -1]

        self._delete_expired_records(token, version, expired_rows)
        return logins

    @staticmethod
    def _get_absences(token):
        URL = 'https://staff.yandex-team.ru/gap-api/api/export_gaps?field=person_login&field=workflow'
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        req = get(URL, headers=headers).json()
        return req['persons'].keys()

    def _set_staff_assignees(self, token):
        URL = 'https://staff-api.yandex-team.ru/v3/persons?_pretty=1&department_group.parent.id=122774\
            &official.is_dismissed=false&_limit=1000&_fields=login,created_at,department_group.department,name,official'
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        req = get(URL, headers=headers).json()

        absences = self._get_absences(token)
        exception_list = self._get_exception_list(token)

        heads = []
        department_heads = list(map(lambda h: h['department_group']['department']['heads'], req['result']))
        for heads_list in department_heads:
            for head in heads_list:
                if head['person']['login'] not in heads:
                    heads.append(head['person']['login'])

        assignees = []
        for assignee in req['result']:
            if assignee['department_group']['department']['name']['full']['ru'] not in DEPARTMENT_EXCEPTIONS:
                login = assignee['login']
                if login not in absences and login not in heads and login not in exception_list:
                    join_at = datetime.strptime(assignee['official']['join_at'], '%Y-%m-%d')
                    work_experience = datetime.now() - join_at
                    if work_experience.days >= 60:
                        assignees.append(login)
        self.assignees = assignees

    def _set_users_workload(self, issues):
        users_workload_ = {}

        for issue in issues:
            if issue.assignee.login in users_workload_.keys():
                users_workload_[issue.assignee.login] = users_workload_[issue.assignee.login] + 1
            else:
                users_workload_[issue.assignee.login] = 1

        for assignee in self.assignees:
            if assignee not in users_workload_.keys():
                users_workload_[assignee] = 0

        self.users_workload = users_workload_

    @property
    def random_assignee(self):
        minimum = min(self.users_workload.values())

        available_assignees = [
            assignee for assignee in self.users_workload.keys()
            if self.users_workload[assignee] == minimum
        ]

        shuffle(available_assignees)
        return choice(available_assignees)

    @staticmethod
    def get_issues(token):
        from startrek_client import Startrek

        client = Startrek(
            useragent='python',
            base_url=STARTREK_URL,
            token=token
        )
        issues = client.issues.find(
            filter={'queue': 'MARKETSHOPSCHECK', 'resolution': 'empty()'},
            per_page=100
        )
        return issues

    def update_issues(self, token):
        issues = self.get_issues(token)
        without_assignee = [issue for issue in issues if issue.assignee is not None]
        headers = {'Authorization': 'OAuth %s' % token}

        self._set_staff_assignees(token)
        self._set_users_workload(without_assignee)

        edited_counter = 0
        for issue in issues:
            if issue.assignee is None:
                url = '%s/issues/%s' % (STARTREK_URL, issue.key)
                data = {'assignee': self.random_assignee}
                patch(url, data=dumps(data, ensure_ascii=False).encode('UTF-8'),
                      headers=headers)

                edited_counter = edited_counter + 1
                self.users_workload[data['assignee']] = self.users_workload[data['assignee']] + 1
                logging.info('%s was assigned to %s as random performer.' % (data['assignee'], issue.key))
            else:
                comments = list(issue.comments.get_all())
                extreme_comment = comments[len(comments)-1]

                if extreme_comment.createdBy.login == 'spock' \
                        and issue.weight != 0.1 and issue.status.display == 'Открыт':
                    comment_createdAt = datetime.strptime(extreme_comment.createdAt,
                                                          '%Y-%m-%dT%H:%M:%S.%f+0000') + timedelta(hours=3)
                    time_range = datetime.now() - comment_createdAt

                    if int(time_range.seconds/3600) >= 3:
                        issue.comments.create(summonees=issue.assignee.login)
                        url = '%s/issues/%s' % (STARTREK_URL, issue.key)
                        data = {'weight': 0.1}
                        patch(url, data=dumps(data, ensure_ascii=False).encode('UTF-8'),
                              headers=headers)

                        edited_counter = edited_counter + 1
                        logging.info('Was made a reminder in %s for %s.' % (issue.key, issue.assignee.login))
        logging.info('Successfully edited %i issues.' % edited_counter)


class DeliveryRandomAssignee(sdk2.Task):
    '''Task selects a random performer to perform a check with delivery.'''

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('requests'),
            environments.PipEnvironment(
                'startrek_client',
                custom_parameters=['requests==2.18.4']
            )
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = \
            '''Task selects a random performer to perform a check with delivery.
Running every hour every day between 9 and 19 o`clock.
            '''
        max_restarts = 3
        dump_disk_usage = False
        fail_on_any_error = True

        oauth_token = sdk2.parameters.YavSecret(
            'OAuth ST token (also used to wiki access)',
            required=True
        )

    @staticmethod
    def is_time_between(current_time):
        start_time = time(9, 0)
        end_time = time(19, 0)

        if current_time > start_time and current_time < end_time:
            return True
        return False

    def on_execute(self):
        from sandbox.projects.MarketQC.delivery_random_assignee.expert_checks_assignee import run_selection
        from sandbox.projects.MarketQC.delivery_random_assignee.manager_escalation import run as run_escalation

        token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]

        current_time = time(datetime.now().hour, datetime.now().minute)
        if datetime.now().isoweekday() < 6 and self.is_time_between(current_time):
            run_selection(st_token=token)
            Assignees_selector().update_issues(token=token)
            run_escalation(token)
