# -*- coding: utf-8 -*-
from json import dumps
import xml.etree.ElementTree as ET
from requests import get, post, patch
from datetime import time, timedelta, datetime as dt

ST_FILTER = 'Queue: BQ Components: 86737 Resolution: empty() Assignee: notEmpty()'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def get_user_staff(login, token):
    headers = {'Authorization': 'OAuth {}'.format(token), 'content-type': 'application/json',
               'Accept-Charset': 'UTF-8'}
    url = 'https://staff-api.yandex-team.ru/v3/persons?_one=1&login={}'.format(login)
    return get(url, headers=headers).json()


def get_previous_update_time(issue):
    updates = []
    for event in filter(lambda e: e.type == 'IssueUpdated', issue.changelog):
        if len(list(filter(lambda f: f['field'].name == 'Количество тестирований', event.fields))) > 0:
            updates.append(dt.strptime(event.updatedAt, '%Y-%m-%dT%H:%M:%S.%f+0000') + timedelta(hours=3))

    return updates[-1]


def get_weekends(start_date, end_date):
    url = 'https://api.calendar.yandex-team.ru/export/holidays.xml?' \
        'start_date={}&end_date={}&country_id=225&out_mode=all'.format(start_date, end_date)

    for day in ET.fromstring(get(url).text).find('get-holidays').find('days').iter('day'):
        if day.attrib['is-holiday'] == '1':
            yield day.attrib['date']


def daterange(start_date, end_date):
    for n in range(int((start_date - end_date).days) + 1):
        yield dt.combine((end_date + timedelta(n)).date(), time(0, 0))


def working_time_difference(event):
    difference = [event] + list(daterange(dt.now(), event))[1:] + [dt.now()]
    weekends = list(get_weekends(difference[0].date(), difference[-1].date()))

    time_range = 0

    for i in range(len(difference)-1):
        cur_date = difference[i].date()
        next_date = difference[i+1].date()

        if str(cur_date) in weekends and str(next_date) in weekends:
            continue
        elif str(cur_date) in weekends:
            if difference[i+1].time() > time(18, 0):
                time_range += 9
            else:
                time_range += difference[i+1].time().hour - 9
        elif str(next_date) in weekends:
            if difference[i].time() < time(9, 0):
                time_range += 9
            else:
                time_range += 18 - difference[i].time().hour
        else:
            start_time = difference[i].time()
            end_time = difference[i+1].time()

            if difference[i].date() == difference[i+1].date():
                if start_time < time(9, 0):
                    start_time = time(9, 0)
                elif start_time > time(18, 0):
                    start_time = time(18, 0)

                if end_time < time(9, 0):
                    end_time = time(9, 0)
                elif end_time > time(18, 0):
                    end_time = time(18, 0)

                time_range += end_time.hour - start_time.hour
            else:
                if start_time < time(9, 0):
                    start_time = time(9, 0)
                elif start_time > time(18, 0):
                    start_time = time(18, 0)

                time_range += 18 - start_time.hour

                if end_time < time(9, 0):
                    end_time = time(9, 0)
                elif end_time > time(18, 0):
                    end_time = time(18, 0)

                time_range += end_time.hour - 9

    return time_range


def get_script(assignee, iterations):
    login = assignee['login']
    name = assignee['name']['first']['ru']
    chief_login = assignee['chief']['login']
    chief_name = assignee['chief']['name']['first']['ru']

    if iterations == 0:
        summonees = [login]
        comment = '{}, привет! '.format(name)
        comment += 'С момента создания задачи прошло более 8 рабочих часов, пожалуйста, обрати на неё внимание. Спасибо!'
    elif iterations == 1:
        summonees = [chief_login]
        comment = '{}, привет! '.format(chief_name)
        comment += 'Я уже призывал исполнителя, но задача до сих пор не была закрыта, пожалуйста,'\
            ' помоги разобраться в причинах задержки с разбором. Спасибо!'
    else:
        summonees = [login, chief_login]
        comment = 'Коллеги, привет! '\
            'С момента создания задачи прошло уже много времени, пожалуйста, подключитесь к её разбору. Спасибо!'

    return comment, summonees


def update_issue(issue, assignee, iterations, token):
    comment, summonees = get_script(assignee, iterations)
    url = 'https://st-api.yandex-team.ru/v2/issues/{}'.format(issue.key)
    headers = {'Authorization': 'OAuth {}'.format(token)}

    comment_data = {'text': comment, 'summonees': summonees}
    fields_data = {'kolicestvoTestirovanij': iterations + 1}

    post(url + '/comments', data=dumps(comment_data, ensure_ascii=False).encode('utf8'), headers=headers)
    patch(url, data=dumps(fields_data, ensure_ascii=False).encode('utf8'), headers=headers)


def run(token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    for issue in client.issues.find(ST_FILTER, per_page=100):
        iterations = issue.kolicestvoTestirovanij if issue.kolicestvoTestirovanij is not None else 0
        assignee = get_user_staff(issue.assignee.login, token=token)

        createdAt = dt.strptime(issue.createdAt, '%Y-%m-%dT%H:%M:%S.%f+0000') + timedelta(hours=3)
        if iterations > 0:
            createdAt = get_previous_update_time(issue)

        working_time_range = working_time_difference(createdAt)
        if working_time_range > 8:
            update_issue(issue, assignee, iterations, token)
