# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from datetime import timedelta, datetime, time

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
YT_TABLE = '//home/market-quality-control/robot-johnny/first_order_tracking'


def read_table(token, proxy='hahn'):
    import yt.wrapper as yt
    client = yt.client.Yt(token=token, proxy=proxy)
    return client.read_table(yt.TablePath(YT_TABLE), format='yson', raw=False)


def write_table(data, token, proxy='hahn'):
    import yt.wrapper as yt
    client = yt.client.Yt(token=token, proxy=proxy)
    client.write_table(yt.TablePath(YT_TABLE, append=True), data, raw=False)


def execute_query(query, config):
    from psycopg2.extras import NamedTupleCursor
    from psycopg2 import connect

    with connect(cursor_factory=NamedTupleCursor, **config).cursor() as cursor:
        cursor.execute(query)
        return cursor.fetchall()


ORDER_TODAY = """
SELECT DISTINCT ON (o.id)
    o.id order_id
    , shop_id
    , CASE
        WHEN delivery_service_id = 1006360 THEN TRUE
        ELSE FALSE
    END is_express
    , created_at
    , o.rgb
FROM orders o
    JOIN order_delivery d ON o.id = d.order_id
WHERE o.fake = FALSE AND o.preorder = FALSE
    AND o.created_at > current_date
    AND shop_id <> 431782
    AND o.status <> 7
    AND o.status <> 2
"""

ORDER_HISTORY = """
SELECT shop_id
    , MIN(created_at) created_at
FROM orders
WHERE shop_id = ANY('{}'::int[])
    AND status <> 7
    AND status <> 2
GROUP BY shop_id
"""


class FirstOrderTracking(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('psycopg2-binary'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = ''
        max_restarts = 2
        kill_timeout = 20 * 60
        fail_on_any_error = True
        dump_disk_usage = False

        st_token = sdk2.parameters.YavSecret('OAuth ST token', required=True)
        yt_token = sdk2.parameters.YavSecret('OAuth YQL token', required=True)

        with sdk2.parameters.Group('Database parameters') as database_parameters:
            host = sdk2.parameters.String('Host', default='man-i9jmlo2jx3thalwg.db.yandex.net', required=True)
            port = sdk2.parameters.Integer('Port', default=6432, required=True)
            dbname = sdk2.parameters.String('Database', required=True)
            user = sdk2.parameters.String('User', required=True)
            password = sdk2.parameters.YavSecret('Password', required=True)
            sslmode = sdk2.parameters.String('sslmode', default='require', required=True)

    @staticmethod
    def create_issue(shop, token):
        from startrek_client import Startrek
        client = Startrek(useragent='python', base_url=STARTREK_URL, token=token)

        summary = 'У партнера (ID: {}) был создан первый заказ'.format(shop.shop_id)
        description = 'Ссылка на партнёра в Або: https://abo.market.yandex-team.ru/partner/{}'.format(shop.shop_id)\
            + '\nСсылка на заказ в Або: https://abo.market.yandex-team.ru/order/{}'.format(shop.order_id)\
            + '\nСсылка на заказ в ЕО: https://ow.market.yandex-team.ru/order/{}'.format(shop.order_id)\
            + '\n\n#|'\
            + '\n|| Номер заказа| %%{}%%||'.format(shop.order_id)\
            + '\n|| Дата создания заказа| %%{}%%||'.format(shop.created_at)\
            + '\n|#'

        tags = 'FBS'
        if shop.rgb == 4:
            tags = 'DBS'
        elif shop.is_express:
            tags = 'FBS_экспресс'

        client.issues.create(
            queue='ORDERS' if shop.rgb != 4 else 'DBSREGULAR',
            summary=summary,
            type={'name': 'Задача'},
            tags=tags,
            description=description,
            components=[105833] if shop.rgb != 4 else [112304]
        )

    @staticmethod
    def is_time_between(current_time):
        start_time = time(9, 0)
        end_time = time(19, 0)

        if current_time > start_time and current_time < end_time:
            return True
        return False

    def on_execute(self):
        from sandbox.projects.MarketQC.first_order_tracking.qtv_complaints import run

        st_token = self.Parameters.st_token.data()
        st_token = st_token[self.Parameters.st_token.default_key]
        yt_token = self.Parameters.yt_token.data()
        yt_token = yt_token[self.Parameters.yt_token.default_key]

        current_time = time(datetime.now().hour, datetime.now().minute)
        # if self.is_time_between(current_time):
        #     config = dict(host=self.Parameters.host,
        #                 port=self.Parameters.port,
        #                 dbname=self.Parameters.dbname,
        #                 user=self.Parameters.user,
        #                 password=self.Parameters.password.data()[self.Parameters.password.default_key],
        #                 sslmode=self.Parameters.sslmode)

        #     shops_past = list(map(lambda k: k['order_id'], read_table(yt_token)))
        #     orders = {}

        #     for record in execute_query(ORDER_TODAY, config):
        #         if record.order_id not in shops_past and (record.is_express or record.rgb == 4):
        #             orders.setdefault(record.shop_id, []).append(record)

        #     query = ORDER_HISTORY.format('{%s}' % ','.join([str(_) for _ in orders]))
        #     shops_newly = []

        #     for record in execute_query(query, config):
        #         today_orders = sorted(orders[record.shop_id], key=lambda k: k.created_at)
        #         if record.created_at == today_orders[0].created_at:
        #             shops_newly.append(today_orders[0])

        #     shops_uploaded = []
        #     for shop in shops_newly:
        #         shops_uploaded.append({'shop_id': shop.shop_id, 'order_id': shop.order_id})
        #         self.create_issue(shop, token=st_token)

        #     write_table(shops_uploaded, token=yt_token)
        run(yt_token, st_token)
