# -*- coding: utf-8 -*-
from json import dumps
from requests import patch, post
from collections import namedtuple

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
ST_QUERY = 'Queue: QTV AND Resolution: empty() AND (Components: 102636 OR Components: 110529) AND "Macro tag": empty()'

YQL = """
USE hahn;
PRAGMA AnsiInForEmptyOrNullableItemsCollections;

$lambda = ($data) -> {{
    $MbiRows = ListMap(String::SplitToList($data, '\n'), ($row) -> {{$items = String::SplitToList($row, '	', 1 AS Limit);
            return AsTuple($items[0], $items[1]) }});
    return ToDict($MbiRows);
}};

SELECT DISTINCT o.shop_id AS shop_id
    , feed_id
    , offer_id
    , shop_name
    , supplier_id
    , supplier_name
    , ware_md5
    , CAST($lambda(mbi)['url'] AS String) AS feed_url
FROM `//home/market/production/indexer/turbo.stratocaster/offers/recent` AS o
    LEFT JOIN `//home/market/production/indexer/datacamp/united/partners` AS p ON o.shop_id = p.shop_id
WHERE ware_md5 IN ({})
"""


def parse_issue(issue):
    try:
        ware_md5 = issue.description.split('\'offerid\': \'')[1].split('\'')[0]
    except IndexError:
        ware_md5 = ''
    return {
        'key': issue.key,
        'ware_md5': ware_md5
    }


def collect_issues(token):
    from startrek_client import Startrek
    client = Startrek(
        useragent='market-quality-control', base_url=STARTREK_URL,
        token=token, timeout=1600
    )

    for issue in client.issues.find(ST_QUERY, perPage=100):
        yield parse_issue(issue)


def execute_query(query, token=None):
    from yql.api.v1.client import YqlClient
    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        pattern = namedtuple('Record', result.column_names)
        for record in result.get_iterator():
            yield pattern(*record)


def create_comment(data, token):
    comment = '#|'\
        + '\n|| ID магазина| %%{}%%||'.format(data['offer']['shop_id'])\
        + '\n|| Название магазина| %%{}%%||'.format(data['offer']['shop_name'])\
        + '\n|| ID фида| (({} {}))||'.format(data['offer']['feed_url'], data['offer']['feed_id'])\
        + '\n|| ID оффера| %%{}%%||'.format(data['offer']['offer_id'])\
        + '\n|| ID поставщика| %%{}%%||'.format(data['offer']['supplier_id'])\
        + '\n|| Название поставщика| %%{}%%||'.format(data['offer']['supplier_name'])\
        + '\n|#'

    url = 'https://st-api.yandex-team.ru/v2/issues/{}/comments'.format(data['key'])
    headers = {'Authorization': 'OAuth {}'.format(token)}
    post(url, data=dumps({'text': comment}, ensure_ascii=False).encode('utf8'), headers=headers)

    url = 'https://st-api.yandex-team.ru/v2/issues/{}'.format(data['key'])
    patch(url, data=dumps({'macroTag': '1', 'idMagazina': data['offer']['shop_id']}, ensure_ascii=False).encode('utf8'), headers=headers)


def run(yt_token, st_token):
    ware_hex = {}
    for issue in collect_issues(token=st_token):
        ware_hex.setdefault(issue['ware_md5'], []).append(issue['key'])

    new_offers = []
    query = YQL.format(','.join(['\'%s\'' % md5 for md5 in ware_hex]))

    for record in execute_query(query, token=yt_token):
        for key in ware_hex[record.ware_md5]:
            new_offers.append({'key': key, 'offer': record._asdict()})

    for issue in new_offers:
        create_comment(issue, token=st_token)
