# -*- coding: utf-8 -*-

import logging
from requests import get
import xml.etree.ElementTree as ET
from datetime import timedelta, datetime

from sandbox import sdk2
from sandbox.sandboxsdk import environments

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
YT_PATH = '//home/clickhouse-kolkhoz/ch_hitinap/skk_processes/inadequate_price/states_table'


class Source:

    @staticmethod
    def get_states_table(yql_token):
        import yt.wrapper as wrapper

        client = wrapper.client.Yt(
            token=yql_token,
            proxy='hahn',
            config={"tabular_data_format": "dsv"}
        )
        table = list(client.read_table(YT_PATH))
        return table

    @staticmethod
    def save_states_table(yql_token, table):
        import yt.wrapper as wrapper

        client = wrapper.client.Yt(
            token=yql_token,
            proxy='hahn',
            config={"tabular_data_format": "dsv"}
        )
        client.write_table('<append=%false>' + YT_PATH, table, raw=False)

    def _set_working_dates(self):
        current_date = datetime.now().strftime('%Y-%m-%d')
        previous_date = (datetime.now() - timedelta(days=30)).strftime('%Y-%m-%d')

        url = 'https://api.calendar.yandex-team.ru/export/holidays.xml?' + \
            'start_date={previous_date}&end_date={current_date}&country_id=225&out_mode=all'.format(
                current_date=current_date,
                previous_date=previous_date
            )

        root = ET.fromstring(get(url).text)
        calendar_tree = list(root.find('get-holidays').find('days').iter('day'))
        calendar_tree.reverse()

        previous_date = datetime.now() - timedelta(days=1)
        for day in calendar_tree:
            if (previous_date - datetime.strptime(day.attrib['date'], '%Y-%m-%d')).days >= 0:
                if int(day.attrib['is-holiday']) != 1:
                    previous_date = day.attrib['date']
                    break

        if type(previous_date) == datetime:
            raise RuntimeError(
                'The number of iterations exceeded the set limit, it seems that something went wrong.')

        self.current_date = datetime.now().strftime('%Y-%m-%dT%H:%M:%S')
        self.previous_date = previous_date

    def update_states_table(self, token, yql_token):
        from startrek_client import Startrek

        client = Startrek(
            useragent='python',
            base_url=STARTREK_URL,
            token=token
        )

        self._set_working_dates()
        states_table = self.get_states_table(yql_token)
        shops = dict(map(lambda row: (row['shop'], states_table.index(row)), states_table))

        issues = client.issues.find(
            filter={'queue': 'TURBOQUALITY', 'components': ['87718', '87717', '87716', '87715'],
                    'resolved': {'from': self.previous_date, 'to': self.current_date}},
            per_page=100
        )

        for issue in issues:
            ticket = {}
            ticket['shop'] = issue.summary.replace('Мониторинг: у партнера PWA (', '')\
                .replace('Мониторинг: у партнера Турбо (', '')\
                .replace(') были найдены низкие цены', '')\
                .replace(') были найдены высокие цены', '')

            ticket['prepared'] = 'False'
            for c in issue.components:
                if 'фиды' in c.display and 'низкая' in c.display:
                    ticket['check_type'] = 'pwa_less'
                elif 'фиды' in c.display and 'высокая' in c.display:
                    ticket['check_type'] = 'pwa_more'
                elif 'Турбо' in c.display and 'низкая' in c.display:
                    ticket['check_type'] = 'turbo_less'
                elif 'Турбо' in c.display and 'высокая' in c.display:
                    ticket['check_type'] = 'turbo_more'

            ticket['last_solution'] = issue.tags[0] if len(issue.tags) > 0 else 'None'

            if ticket['shop'] in shops.keys():
                filter_ = [shops[shop]
                           for shop in shops.keys() if shop == ticket['shop']]

                for i in filter_:
                    if ticket['check_type'] == states_table[i]['check_type']:
                        if 'мимо' in ticket['last_solution'] and 'мимо' in states_table[i]['last_solution']:
                            ticket['whitelist'] = (
                                datetime.now() + timedelta(days=180)).strftime('%Y-%m-%d')
                        elif 'мимо' in ticket['last_solution']:
                            ticket['whitelist'] = (
                                datetime.now() + timedelta(days=90)).strftime('%Y-%m-%d')
                        elif 'попал' in ticket['last_solution']:
                            ticket['whitelist'] = (
                                datetime.now() + timedelta(days=30)).strftime('%Y-%m-%d')
                        else:
                            ticket['whitelist'] = (
                                datetime.now() + timedelta(days=2)).strftime('%Y-%m-%d')
                        states_table[i] = ticket
                        break
            else:
                if 'мимо' in ticket['last_solution']:
                    ticket['whitelist'] = (
                        datetime.now() + timedelta(days=90)).strftime('%Y-%m-%d')
                elif 'попал' in ticket['last_solution']:
                    ticket['whitelist'] = (
                        datetime.now() + timedelta(days=30)).strftime('%Y-%m-%d')
                else:
                    ticket['whitelist'] = (
                        datetime.now() + timedelta(days=2)).strftime('%Y-%m-%d')
                states_table.append(ticket)

        for row in states_table:
            whitelist = datetime.strptime(row['whitelist'], '%Y-%m-%d')
            if (datetime.now() - whitelist).days >= 0:
                row['prepared'] = 'True'

        opened_issues = client.issues.find(
            filter={'queue': 'TURBOQUALITY',
                    'components': ['87718', '87717', '87716', '87715'], 'resolution': 'empty()'},
            per_page=100
        )

        for issue in opened_issues:
            shop = issue.summary.replace('Мониторинг: у партнера PWA (', '')\
                .replace('Мониторинг: у партнера Турбо (', '')\
                .replace(') были найдены низкие цены', '')\
                .replace(') были найдены высокие цены', '')

            for c in issue.components:
                if 'фиды' in c.display and 'низкая' in c.display:
                    check_type = 'pwa_less'
                elif 'фиды' in c.display and 'высокая' in c.display:
                    check_type = 'pwa_more'
                elif 'Турбо' in c.display and 'низкая' in c.display:
                    check_type = 'turbo_less'
                elif 'Турбо' in c.display and 'высокая' in c.display:
                    check_type = 'turbo_more'

            for row in states_table:
                if row['shop'] == shop and row['check_type'] == check_type:
                    row['prepared'] = 'False'
                    break

        self.save_states_table(yql_token, states_table)
        self.states_table = states_table

    @staticmethod
    def run_query(yql_token, offers_type, condition):
        from yql.api.v1.client import YqlClient

        client = YqlClient(token=yql_token, db='hahn')
        request = client.query('''
pragma yt.QueryCacheMode = 'disable';
$base = (
select shop_id, price, previous_price, url, feed_url
  from (
    select
        source, shop_id, url, feed_url, type,
        Yson::ConvertToDouble(Yson::Lookup(Yson::Parse(t.price), "current"), Yson::Options(false as Strict, true as AutoConvert)) as price,
        Yson::ConvertToDouble(Yson::Lookup(Yson::Parse(t.price), "previous"), Yson::Options(false as Strict, true as AutoConvert)) as previous_price
    from `//home/turborss/production/offers/offer_base` as t
  )
  where
    price {condition}
    and source = '{offers_type}'
    and type = 'offer'
);

$samples = (
  select
    shop_id,
    aggregate_list_distinct(
      AsStruct(
        url as url,
        feed_url as feed_url,
        price as price,
        previous_price as previous_price
      )
    ) as samples
  from (
    select
      shop_id, url, feed_url, price, previous_price,
      row_number() over w as row_no
    from $base
    window w as (
      partition by shop_id
      order by price desc
    )
  )
  where row_no < 5
  group compact by shop_id
);

$stats = (
  select shop_id, count(*) as suspecious_offers
  from $base
  group by shop_id
);

select
  stats.shop_id as shop_id,
  stats.suspecious_offers as suspecious_offers,
  samples.samples as samples
from $stats as stats
  inner join $samples as samples using (shop_id)
order by suspecious_offers desc;
        '''.format(condition=condition, offers_type=offers_type), syntax_version=1)
        request.run()

        table = []
        for result in request.get_results():
            column_names = ['shop', 'suspecious_offers', 'samples']
            for row in result.get_iterator():
                table.append({column_names[row.index(cell)]: cell if row.index(cell) != 2
                              else list(map(lambda c: c.__dict__, cell)) for cell in row})
        return table

    def _get_high_prices(self, yql_token, offers_type, cutoff):
        condition = '> ' + cutoff
        return self.run_query(yql_token, offers_type, condition)

    def _get_low_prices(self, yql_token, offers_type, cutoff):
        condition = '< ' + cutoff
        return self.run_query(yql_token, offers_type, condition)

    def uploading_to_startrek(self, token, hypotheses, specific_tag):
        from startrek_client import Startrek

        client = Startrek(
            useragent='python',
            base_url=STARTREK_URL,
            token=token
        )

        shops = dict(map(lambda s: (s['shop'], {'index': self.states_table.index(
            s), 'prepared': s['prepared'], 'check_type': s['check_type']}), self.states_table))

        if specific_tag == 'pwa_less':
            params = ('PWA', 'низкие', 'Бесплатные фиды - низкая цена')
        elif specific_tag == 'pwa_more':
            params = ('PWA', 'высокие', 'Бесплатные фиды - высокая цена')
        elif specific_tag == 'turbo_less':
            params = ('Турбо', 'низкие', 'Турбо - низкая цена')
        elif specific_tag == 'turbo_more':
            params = ('Турбо', 'высокие', 'Турбо - высокая цена')
        name, level, component = params

        firstly = []
        secondly = []
        for row in hypotheses:
            if row['shop'] in shops.keys():
                if shops[row['shop']]['prepared'] != 'False' \
                        and shops[row['shop']]['check_type'] == specific_tag:
                    row['index'] = shops[row['shop']]['index']
                    secondly.append(row)
            else:
                row['index'] = 'None'
                firstly.append(row)
        ready_to_check = [row for row in firstly]
        for row in secondly:
            ready_to_check.append(row)

        quantity = 80 if len(ready_to_check) > 80 else len(ready_to_check)
        for i in range(quantity):
            row = ready_to_check[i]
            summary = 'Мониторинг: у партнера {name} ({shop}) были найдены {level} цены'.format(
                name=name,
                shop=row['shop'],
                level=level
            )

            if 'turbo' in specific_tag:
                ban_url = 'Таблица бана: https://abo.market.yandex-team.ru/turbo/shop/ban'
            elif 'pwa' in specific_tag:
                ban_url = \
                    'Админка приложений: https://turboapps.yandex.ru/console/applications#page-size=15&page-number=0&filter-field=title:%s' % row[
                        'shop']

            samples = {}
            for sample in row['samples']:
                if sample['feed_url'] not in samples.keys():
                    samples[sample['feed_url']] = [
                        {'url': sample['url'], 'price': sample['price']}]
                else:
                    samples[sample['feed_url']].append(
                        {'url': sample['url'], 'price': sample['price']})

            description = '{ban}\n\nКоличество подозрительных предложений: **{suspecious_offers}**\n\n'.format(
                ban=ban_url,
                suspecious_offers=row['suspecious_offers']
            )
            for feed in samples.keys():
                paragraph = 'Адрес фида: {feed}\n#| || **Предложения магазина:**| **Цена в фиде:**||'.format(
                    feed=feed)
                for offer in samples[feed]:
                    paragraph = paragraph + \
                        '|| {url}| {price}||'.format(
                            url=offer['url'], price=offer['price'])
                paragraph = paragraph + ' |#\n'
                description = description + paragraph

            client.issues.create(
                queue='TURBOQUALITY',
                summary=summary,
                type={'name': 'Задача'},
                description=description,
                components=component
            )
        logging.info('{quantity} tasks were successfully uploaded to ST with the {tag} tag'.format(
            quantity=quantity,
            tag=specific_tag
        ))

    def get_started(
        self,
        turbo_upper_cutoff,
        turbo_lower_cutoff,
        pwa_upper_cutoff,
        pwa_lower_cutoff,
        yql_token,
        token
    ):
        self.update_states_table(token, yql_token)

        self.uploading_to_startrek(hypotheses=self._get_high_prices(
            yql_token=yql_token, offers_type='pwa', cutoff=str(pwa_upper_cutoff)), token=token, specific_tag='pwa_more')
        self.uploading_to_startrek(hypotheses=self._get_low_prices(
            yql_token=yql_token, offers_type='pwa', cutoff=str(pwa_lower_cutoff)), token=token, specific_tag='pwa_less')
        self.uploading_to_startrek(hypotheses=self._get_high_prices(
            yql_token=yql_token, offers_type='turbo+', cutoff=str(turbo_upper_cutoff)), token=token, specific_tag='turbo_more')
        self.uploading_to_startrek(hypotheses=self._get_low_prices(
            yql_token=yql_token, offers_type='turbo+', cutoff=str(turbo_lower_cutoff)), token=token, specific_tag='turbo_less')


class InadequatePricesCheck(sdk2.Task):
    '''Task unloads stores for checking for inadequate prices.'''

    class Requirements(sdk2.Requirements):
        ram = 2048
        cores = 2
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
            environments.PipEnvironment(
                'startrek_client',
                custom_parameters=['requests==2.18.4']
            )
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Task unloads stores for checking for inadequate prices.'
        max_restarts = 3
        dump_disk_usage = False
        fail_on_any_error = True

        oauth_token = sdk2.parameters.YavSecret(
            'OAuth ST token',
            required=True
        )
        yql_token = sdk2.parameters.YavSecret(
            'OAuth YQL token',
            required=True
        )

        with sdk2.parameters.Group('Turbo') as turbo:
            turbo_upper_cutoff = sdk2.parameters.Integer(
                'Upper bound of the condition', default=7499999)
            turbo_lower_cutoff = sdk2.parameters.Integer(
                'Lower bound of the condition', default=2)

        with sdk2.parameters.Group('PWA') as pwa:
            pwa_upper_cutoff = sdk2.parameters.Integer(
                'Upper bound of the condition', default=9999999)
            pwa_lower_cutoff = sdk2.parameters.Integer(
                'Lower bound of the condition', default=2)

    def on_execute(self):
        token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]
        yql_token = self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key]

        Source().get_started(
            turbo_upper_cutoff=self.Parameters.turbo_upper_cutoff,
            turbo_lower_cutoff=self.Parameters.turbo_lower_cutoff,
            pwa_upper_cutoff=self.Parameters.pwa_upper_cutoff,
            pwa_lower_cutoff=self.Parameters.pwa_lower_cutoff,
            yql_token=yql_token,
            token=token
        )
