# -*- coding: utf-8 -*-
import functools
import re
import os

from json import loads
from datetime import datetime, timedelta
from dateutil.parser import parse as parseIso8601

from sandbox.projects.MarketQC.orders_without_final_status.lib.problem_type import spread_unbound_orders
from sandbox.projects.MarketQC.orders_without_final_status.lib.update_issues import update_issue, close_outdated_issues

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
YT_TABLE_PATH = '//home/market-quality-control/monofmon/orders_without_final_status'

ST_FILTER = {
    'resolution': 'empty()',
    'queue': [
        'MQMONDEMAND',
        'MQMDELIVERYDSDO',
        'MQMSHIPMENTDO',
        'MQMDELIVERYDOSC',
        'MQMLASTMILE',
        'MQMCREATE',
        'MQMSHIPMENTDS',
        'MQMDELIVERYFFSD',
        'MQMDELIVERYMK',
        'MQMDELIVERYDSSD',
        'MQMDELIVERYDSSC',
        'MQMDELIVERYSCSD',
        'MQMFINALSTATUS',
        'MQMSHIPMENTSC',
        'MQMASSEMBLYFF',
        'MQMSHIPMENTFF',
        'MQMDELIVERYFFSC',
        'MQMDELIVERYSCSC'
    ]
}


def write_table(table_write_path, table_data, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)

    if table_write_path not in yt.search(table_write_path.rpartition('/')[0], node_type=['table'],
                                         client=client):
        with open(os.path.join(os.path.dirname(__file__), 'schema.json'), 'r') as f:
            schema = loads(f.read())

        client.create('table', table_write_path, attributes={'schema': schema})

    client.write_table(yt.TablePath(table_write_path, append=False), table_data,
                       raw=False)


def process_issues(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        orders_data = {}
        for issue in func(*args, **kwargs):
            if issue.customerOrderNumber is not None:
                for order_number in re.split(r',|;|\s+|\\|/|-', issue.customerOrderNumber):
                    order_number = re.sub(r'[^0-9]', '', order_number)
                    if order_number.isdigit():
                        orders_data.setdefault(int(order_number), []).append(issue.key)

        return orders_data
    return wrapper


def handle_table_data(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return {x['order_id']: x for x in func(*args, **kwargs) if x['problem_type'] > 0}

    return wrapper


@process_issues
def collect_issues(filter_, token=None):
    from startrek_client import Startrek

    client = Startrek(useragent='market-quality-control', base_url=STARTREK_URL,
                      token=token, timeout=1600)

    return client.issues.find(filter=filter_, perScroll=100, scrollType='sorted',
                              scrollTTLMillis=20000)


@handle_table_data
def read_table(table_read_path, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)

    return client.read_table(yt.TablePath(table_read_path), format='json', raw=False)


@handle_table_data
def execute_query(query, token=None):
    from yql.api.v1.client import YqlClient

    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        column_names = result.column_names
        for record in result.get_iterator():
            yield {column_names[i]: val for i, val in enumerate(record)}


def find_opened_issues(orders_data, st_token=None, yt_token=None):
    opened_issues = collect_issues(ST_FILTER, token=st_token)
    today = datetime.now().date().strftime('%Y-%m-%d')

    without_binding, relevant_issues, outdated_issues = {}, {}, {}
    for k, v in orders_data.items():
        if v['tickets'] is None and v['actual']:
            if k in opened_issues:
                v.update({'tickets': ', '.join(opened_issues[k]),
                          'is_ticket_created_by_robot': False,
                          'updated_at': today})
            else:
                without_binding.setdefault(v['problem_type'], []).append(v)

        elif v['is_ticket_created_by_robot'] and v['actual']:
            relevant_issues.setdefault(v['tickets'], []).append(v)

        elif v['is_ticket_created_by_robot'] and not v['actual'] and \
                datetime.now() - parseIso8601(v['updated_at']) < timedelta(days=30):
            outdated_issues.setdefault(v['tickets'], []).append(v)

    with open(os.path.join(os.path.dirname(__file__), 'queues.json'), 'r') as f:
        queues = loads(f.read())

    # problem_type == 10000
    if 10 in without_binding:
        dbs_merch = {}
        for partner in without_binding[10]:
            dbs_merch.setdefault(partner['problem_partner'], []).append(partner)

        dangerous_orders = []
        regular_orders = []

        for k in dbs_merch:
            if len(dbs_merch[k]) > 2:
                dangerous_orders.extend(dbs_merch[k])
            else:
                regular_orders.extend(dbs_merch[k])

        without_binding[10] = regular_orders
        if len(dangerous_orders) > 0:
            without_binding[10000] = dangerous_orders

    for k in without_binding:
        try:
            orders_data.update(
                spread_unbound_orders(
                    without_binding[k], queues[str(k)], st_token
                )
            )
            if k == 10:
                spread_unbound_orders(
                    without_binding[k], queues['10.1'], st_token
                )
        except KeyError:
            continue

    for v in orders_data.values():
        if 'contacts' in v:
            v.pop('contacts')

    write_table(YT_TABLE_PATH, list(orders_data.values()), token=yt_token)

    map(lambda k: update_issue(k, relevant_issues[k], st_token=st_token), relevant_issues)
    map(lambda k: close_outdated_issues(k, outdated_issues[k], st_token=st_token), outdated_issues)


def run(db_config, st_token=None, yt_token=None):
    orders_data = read_table(YT_TABLE_PATH, token=yt_token)
    today = datetime.now().date().strftime('%Y-%m-%d')

    with open(os.path.join(os.path.dirname(__file__), 'query.yql'), 'r') as f:
        yql_query = f.read()

    pending_orders = execute_query(yql_query.format(**db_config), token=yt_token)
    for k in set(pending_orders).difference(orders_data):
        pending_orders[k].update({'actual': True,
                                  'created_at': today,
                                  'updated_at': today,
                                  'tickets': None,
                                  'is_ticket_created_by_robot': None})
        orders_data[k] = pending_orders[k]

    for order_id, order_data in pending_orders.items():
        if order_id in orders_data:
            if order_data['problem_type'] != orders_data[order_id]['problem_type']:
                orders_data[order_id]['problem_type'] = order_data['problem_type']
                orders_data[order_id]['updated_at'] = today
            orders_data[order_id]['contacts'] = order_data['contacts']

    map(lambda k: orders_data[k].update({'actual': False, 'updated_at': today}),
        set(orders_data).difference(pending_orders))

    find_opened_issues(orders_data, st_token=st_token, yt_token=yt_token)
