# -*- coding: utf-8 -*-
from datetime import datetime
import functools
import logging
import os

ISO_WEEKDAYS = ('MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT', 'SUN')
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def is_condition_met(expression):
    if expression == '*':
        return True
    elif expression == 'WEEKDAY':
        if datetime.now().isoweekday() < 6:
            return True
        else:
            return False
    elif datetime.now().isoweekday() == ISO_WEEKDAYS.index(expression) + 1:
        return True
    else:
        return False


def create_excel_file(orders):
    from pandas import DataFrame

    file_name = '{}/{}.csv'.format(os.getcwd(), datetime.now().date())

    df = DataFrame(orders)
    df = df.drop(['actual', 'autocancel_flag', 'problem_partner', 'problem_type', 'tickets'], axis=1)
    if 'contacts' in df.columns:
        df.drop(['contacts'], axis=1)

    df.to_csv(file_name, index=False, compression='gzip', encoding='utf-8')
    return file_name


def logging_exception(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        from startrek_client.exceptions import UnprocessableEntity
        from startrek_client.exceptions import BadRequest
        from startrek_client.exceptions import Forbidden
        from startrek_client.exceptions import NotFound

        try:
            return func(*args, **kwargs)
        except Forbidden:
            logging.error('Failed to create issue due to access error.')
        except UnprocessableEntity:
            logging.error('An incorrect issue type was selected.')
        except BadRequest:
            logging.error('Invalid request parameters.')
        except IOError:
            logging.error('Unable to open file for reading.')
        except NotFound:
            logging.error('No transition was found.')

    return wrapper


@logging_exception
def create_issue(queue, summary, tags, description, type, orders_insertion_type,
                 orders, components=None, problem_partner=None, st_token=None):
    from startrek_client import Startrek

    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=st_token)

    attachments = []
    summary = summary.format(problem_partner=problem_partner)
    description = description.format(problem_partner=problem_partner)
    tags = [tag.format(problem_partner=str(problem_partner).replace(',', '').replace(';', ''))
            for tag in tags]

    email = ''
    contacts = list(filter(lambda obj: 'contacts' in obj, orders))
    if len(contacts) > 0:
        email = contacts[0]['contacts']['email'] if contacts[0]['contacts'] is not None else ''

    if orders_insertion_type == 'excel':
        file_name = create_excel_file(orders)
        attachments.append(file_name)
    else:
        description += '\n'.join([str(_['order_id']) for _ in orders])

    issue = client.issues.create(
        queue=queue,
        summary=summary,
        type=type,
        description=description,
        tags=tags,
        components=components,
        attachments=attachments,
        customerOrderNumber=', '.join(
            [str(_['order_id']) for _ in orders]),
        defectOrders=len(orders),
        customerEmail=email
    )

    return issue.key


def groupby_problem_partner(orders):
    problem_partner = {}
    for order_data in orders:
        problem_partner.setdefault(str(order_data['problem_partner']), []).append(order_data)

    return problem_partner


def spread_unbound_orders(orders, instructions, st_token=None):
    today = datetime.now().date().strftime('%Y-%m-%d')

    grouping_type = instructions['grouping_type']
    orders_insertion_type = instructions['orders_insertion_type']
    schedule = instructions['schedule']
    startrek = instructions['startrek']

    if is_condition_met(schedule):
        if grouping_type == 'problem_partner':
            orders = groupby_problem_partner(orders)
            for problem_partner in orders:
                issue_key = create_issue(orders_insertion_type=orders_insertion_type,
                                         problem_partner=problem_partner,
                                         st_token=st_token,
                                         orders=orders[problem_partner],
                                         **startrek)
                if issue_key is not None:
                    for order in orders[problem_partner]:
                        order.update({'tickets': issue_key,
                                      'updated_at': today,
                                      'is_ticket_created_by_robot': True})
            orders = [order for suborders in orders.values()
                      for order in suborders]
        else:
            issue_key = create_issue(orders_insertion_type=orders_insertion_type,
                                     orders=orders,
                                     st_token=st_token,
                                     **startrek)
            if issue_key is not None:
                for order in orders:
                    order.update({'tickets': issue_key,
                                  'updated_at': today,
                                  'is_ticket_created_by_robot': True})

    return {order['order_id']: order for order in orders}
