# -*- coding: utf-8 -*-
import re
import functools
from json import dumps
from requests import patch

from sandbox.projects.MarketQC.orders_without_final_status.lib.problem_type import logging_exception
from sandbox.projects.MarketQC.orders_without_final_status.lib.problem_type import create_excel_file

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def parse_issue_orders(issue):
    orders = []
    if issue.customerOrderNumber is not None:
        for order_number in re.split(r',|;|\s+|\\|/|-', issue.customerOrderNumber):
            order_number = re.sub(r'[^0-9]', '', order_number)
            if order_number.isdigit():
                orders.append(int(order_number))

    return orders


def retry_transaction(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        from startrek_client.exceptions import NotFound

        try:
            return func(*args, **kwargs)
        except NotFound:
            return func(transition='closed', *args, **kwargs)

    return wrapper


def update_customer_orders(issue, orders, st_token=None):
    url = 'https://st-api.yandex-team.ru/v2/issues/{}'.format(issue.key)
    headers = {'Authorization': 'OAuth {}'.format(st_token)}

    patch(url, data=dumps({'customerOrderNumber': orders}, ensure_ascii=False).encode('UTF-8'),
          headers=headers)


@logging_exception
@retry_transaction
def close_outdated_issues(st_ticket, orders, transition='close', st_token=None):
    from startrek_client import Startrek

    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=st_token)

    issue = client.issues[st_ticket]
    issue_orders = parse_issue_orders(issue)

    if len(issue_orders) > 0 and 'MARKETQUALITY' not in issue.key:
        if issue.resolution is None and len(set(issue_orders).difference([_['order_id'] for _ in orders])) == 0:
            comment = 'Все заказы, представленные в тикете, более не актуальны. Тикет автоматически закрыт.'
            issue.transitions[transition].execute(comment=comment, resolution='fixed')


@logging_exception
def update_issue(st_ticket, orders, st_token=None):
    from startrek_client import Startrek

    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=st_token)

    issue = client.issues[st_ticket]
    issue_orders = parse_issue_orders(issue)

    if len(issue_orders) > 0 and 'MARKETQUALITY' not in issue.key:
        if issue.resolution is not None:
            assignee = issue.assignee.login if issue.assignee is not None else None
            comment = 'Привет! Задача закрыта, но в ней еще остались актуальные заказы, пожалуйста, вернитесь к её разбору.'

            issue.comments.create(text=comment, summonees=assignee)

        if len(issue_orders) > len(orders):
            actual_orders = [order['order_id'] for order in orders]
            update_customer_orders(issue, orders=', '.join([str(_) for _ in actual_orders]),
                                   st_token=st_token)

            comment = 'Информация в тикете была автоматически изменена. Актуальный список заказов во вложении.'
            attachments = create_excel_file(orders)
            issue.comments.create(text=comment, attachments=[attachments])
