# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals
from collections import namedtuple
from datetime import timedelta, datetime as dt

from sandbox import sdk2
from sandbox.sandboxsdk import environments

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'

YQL = """
$yesterday = CAST(CurrentUtcDate() - DateTime::IntervalFromDays(1) AS String);
$callable = ($table_name) -> { return $table_name >= $yesterday };

SELECT DISTINCT domain
    , feed_url
    , business_id
    , rules
FROM hahn.FILTER(`//home/antifraud/export/market/shops/fast/tv_fakes`, $callable)
WHERE ListLength(rules) > 0;
"""

DAILY = """
USE hahn;
$yesterday = CAST(CurrentUtcDate() - DateTime::IntervalFromDays(1) AS String);
$table = '//home/antifraud/export/market/shops/daily/tv_fakes/' || $yesterday;

SELECT DISTINCT domain
    , shop_id
    , rules
FROM $table WHERE ListLength(rules) > 0;
"""

DIRECT = """
USE hahn;
$yesterday = CAST(CurrentUtcDate() - DateTime::IntervalFromDays(1) AS String);
$table = '//home/antifraud/export/market/shops/daily/direct_fakes/' || $yesterday;

SELECT DISTINCT domain
    , shop_id
    , rules
FROM $table WHERE ListLength(rules) > 0;
"""


def execute_query(query, token):
    from yql.api.v1.client import YqlClient
    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        pattern = namedtuple('Record', result.column_names)
        for record in result.get_iterator():
            yield pattern(*record)


def collect_previous_issues(token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    ago7d = (dt.now() - timedelta(days=7)).date()
    filter_ = {'queue': 'QTV', 'components': ['105772', '107259'],
               'created': {'from': str(ago7d)}}

    previous_hosts = []
    for issue in client.issues.find(filter=filter_, perPage=100):
        previous_hosts.append(issue.merchant)

    return previous_hosts


class FastTvAntifraud(sdk2.Task):

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = ''
        max_restarts = 2
        kill_timeout = 20 * 60
        fail_on_any_error = True
        dump_disk_usage = False

        st_token = sdk2.parameters.YavSecret('OAuth ST token', required=True)
        yt_token = sdk2.parameters.YavSecret('OAuth YQL token', required=True)

    @staticmethod
    def create_issue(host, token):
        from startrek_client import Startrek
        client = Startrek(useragent='market-quality-control',
                          base_url=STARTREK_URL, token=token)

        summary = 'Антифрод нашел потенциального мошенника {}'.format(host.domain)
        description = 'ID бизнеса: {}'.format(host.business_id)\
            + '\nАдрес фида: {}'.format(host.feed_url)\
            + '\n\nПодробности оценки Антифрода:\n'\
            + ', '.join(host.rules)

        client.issues.create(
            queue='QTV',
            summary=summary,
            type={'name': 'Задача'},
            description=description,
            components=[105772],
            merchant=host.domain
        )

    @staticmethod
    def create_daily_issue(host, token, tags=None):
        from startrek_client import Startrek
        client = Startrek(useragent='market-quality-control',
                          base_url=STARTREK_URL, token=token)

        summary = 'Антифрод нашел потенциального мошенника {}'.format(host.domain)
        description = 'ID магазина: {}'.format(host.shop_id)\
            + '\n\nПодробности оценки Антифрода:\n'\
            + ', '.join(host.rules)

        client.issues.create(
            queue='QTV',
            summary=summary,
            type={'name': 'Задача'},
            description=description,
            components=[107259],
            merchant=host.domain,
            idMagazina=host.shop_id,
            tags=tags
        )

    def on_execute(self):
        from sandbox.projects.MarketQC.search_verticals.tv_fakes.unisearch_fakes import run

        st_token = self.Parameters.st_token.data()
        st_token = st_token[self.Parameters.st_token.default_key]
        yt_token = self.Parameters.yt_token.data()
        yt_token = yt_token[self.Parameters.yt_token.default_key]

        previous_hosts = collect_previous_issues(st_token)
        for record in execute_query(YQL, token=yt_token):
            if record.domain not in previous_hosts:
                self.create_issue(record, token=st_token)

        for record in execute_query(DAILY, token=yt_token):
            if record.domain not in previous_hosts:
                self.create_daily_issue(record, token=st_token)

        for record in execute_query(DIRECT, token=yt_token):
            if record.domain not in previous_hosts:
                self.create_daily_issue(record, token=st_token, tags='Директ')

        run(st_token, yt_token)
