# -*- coding: utf-8 -*-
from collections import namedtuple
from datetime import timedelta, datetime as dt

STARTREK_URL = 'https://st-api.yandex-team.ru/v2'

YQL = """
$yesterday = CAST(CurrentUtcDate() - DateTime::IntervalFromDays(1) AS String);
$callable = ($table_name) -> { return $table_name >= $yesterday };

SELECT DISTINCT domain
    , feed_url
    , rules
FROM hahn.FILTER(`//home/antifraud/export/market/shops/fast/unisearch_fakes`, $callable)
WHERE ListLength(rules) > 0;
"""

DAILY = """
USE hahn;
$yesterday = CAST(CurrentUtcDate() - DateTime::IntervalFromDays(1) AS String);
$table = '//home/antifraud/export/market/shops/daily/unisearch_fakes/' || $yesterday;

SELECT DISTINCT domain
    , feed_url
    , rules
FROM $table WHERE ListLength(rules) > 0;
"""


def execute_query(query, token):
    from yql.api.v1.client import YqlClient
    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        pattern = namedtuple('Record', result.column_names)
        for record in result.get_iterator():
            yield pattern(*record)


def collect_previous_issues(token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    ago7d = (dt.now() - timedelta(days=7)).date()
    filter_ = {'queue': 'US', 'components': ['106834', '107258'], 'created': {'from': str(ago7d)}}

    previous_hosts = []
    for issue in client.issues.find(filter=filter_, perPage=100):
        previous_hosts.append(issue.merchant)

    return previous_hosts


def create_issue(host, token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    summary = 'Антифрод нашел потенциального мошенника {}'.format(host.domain)
    description = '\nАдрес фида: {}'.format(host.feed_url)\
        + '\n\nПодробности оценки Антифрода:\n'\
        + ', '.join(host.rules)

    client.issues.create(
        queue='US',
        summary=summary,
        type={'name': 'Задача'},
        description=description,
        components=[106834],
        merchant=host.domain
    )


def create_daily_issue(host, token):
    from startrek_client import Startrek
    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=token)

    summary = 'Антифрод нашел потенциального мошенника {}'.format(host.domain)
    description = '\nАдрес фида: {}'.format(host.feed_url)\
        + '\n\nПодробности оценки Антифрода:\n'\
        + ', '.join(host.rules)

    client.issues.create(
        queue='US',
        summary=summary,
        type={'name': 'Задача'},
        description=description,
        components=[107258],
        merchant=host.domain
    )


def run(st_token, yt_token):
    previous_hosts = collect_previous_issues(st_token)
    for record in execute_query(YQL, token=yt_token):
        if record.domain not in previous_hosts:
            create_issue(record, token=st_token)

    for record in execute_query(DAILY, token=yt_token):
        if record.domain not in previous_hosts:
            create_daily_issue(record, token=st_token)
