# -*- coding: utf-8 -*-
from __future__ import absolute_import, division

from collections import OrderedDict
from datetime import datetime
import functools

from random import shuffle, choice
from requests import post, get
from time import sleep
from json import dumps

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.projects.MarketQC.sitemarking_daily_uploads.description import DESCRIPTION
from sandbox.projects.MarketQC.sitemarking_daily_uploads.description_modadvert import ADVERT_DESCRIPTION

YT_PROXY = 'hahn'
YT_READ_PATH = '//home/antispam/export/skk/markup_queue/'
YT_AMNESTY = '//home/antispam/export/skk/markup_queue/amnesty'

GRID = 'users/sefomin/sitemarkingexceptions'
MODADVERT_GRID = 'users/hitinap/ispolniteli---razmetka-moderacii-reklamy'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
LIMIT = 15


class SitemarkingDailyUploads(sdk2.Task):
    """Task creates tickets for site markup based on YT table."""

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('requests'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Task creates tickets for site markup based on YT table.'
        max_restarts = 3
        dump_disk_usage = False
        fail_on_any_error = True

        oauth_token = sdk2.parameters.YavSecret('OAuth ST token', required=True)
        yql_token = sdk2.parameters.YavSecret('OAuth YQL token', required=True)

    @staticmethod
    def read_table(table_read_path, token=None, proxy='hahn', end_index=400, columns=['owner', 'full_check', 'base_check']):
        import yt.wrapper as wrapper
        client = wrapper.client.Yt(token=token, proxy=proxy)
        return client.read_table(wrapper.TablePath(table_read_path, end_index=end_index, columns=columns),
                                 format=wrapper.YsonFormat(format='pretty'), raw=False)

    @staticmethod
    def clear_table(table_write_path, table=[], token=None, proxy='hahn'):
        import yt.wrapper as yt
        client = yt.client.Yt(token=token, proxy=proxy)
        client.write_table(yt.TablePath(table_write_path, append=False), table, raw=False)

    @staticmethod
    def collect_issues(filter_, token=None):
        from startrek_client import Startrek

        client = Startrek(useragent='python',
                          base_url=STARTREK_URL, token=token)

        return client.issues.find(filter=filter_, perScroll=1000,
                                  scrollType='unsorted', scrollTTLMillis=20000)

    @staticmethod
    def get_staff_absences(token=None):
        URL = 'https://staff.yandex-team.ru/gap-api/api/export_gaps?field=person_login&field=workflow'
        headers = {'content-type': 'application/json', 'Accept-Charset': 'UTF-8',
                   'Authorization': 'OAuth %s' % token}

        response = get(URL, headers=headers).json()
        return list(response['persons'])

    @staticmethod
    def remove_expired_rows(rows, version, token=None):
        URL = 'https://wiki-api.yandex-team.ru/_api/frontend/%s/.grid/change' % GRID
        headers = {'content-type': 'application/json', 'Accept-Charset': 'UTF-8',
                   'Authorization': 'OAuth %s' % token}

        data = {'version': version,
                'changes': [{'removed_row': {'id': row_id}} for row_id in rows]}

        post(URL, data=dumps(data, ensure_ascii=False).encode('UTF-8'), headers=headers)

    def get_available_assignees(self, grid=GRID, token=None):
        URL = 'https://wiki-api.yandex-team.ru/_api/frontend/%s/.grid' % grid
        headers = {'content-type': 'application/json', 'Accept-Charset': 'UTF-8',
                   'Authorization': 'OAuth %s' % token}
        response = get(URL, headers=headers).json()

        version = response['data']['version']
        wiki_table = response['data']['rows']

        absences = self.get_staff_absences(token)
        expired_rows = []
        available_assignees = []

        for row in wiki_table:
            columns = (row[0]['raw'], row[1]['raw'], row[2]['raw'])
            if (columns[2] != '') and (datetime.now() - datetime.strptime(columns[2], '%Y-%m-%d').days > -1):
                expired_rows.append(row[2]['row_id'])
            if (len(columns[0]) > 0) and (columns[0][0] not in absences):
                available_assignees.extend(columns[0])

        self.remove_expired_rows(expired_rows, version, token)
        return available_assignees

    def get_users_workload(self, components='empty()', token=None):
        filter_ = {'queue': 'SITEMARKING', 'resolution': 'empty()',
                   'assignee': 'notEmpty()', 'components': components}

        workload = OrderedDict()
        for issue in self.collect_issues(filter_, token):
            if issue.assignee.login in workload:
                workload[issue.assignee.login] += 1
            else:
                workload[issue.assignee.login] = 1

        return workload

    def backlog_unique(self, yt_token=None, st_token=None):
        filter_ = {'queue': 'SITEMARKING', 'tags': 'Бэклог'}

        checked_hosts = []
        for issue in self.collect_issues(filter_, st_token):
            checked_hosts.append(issue['6151d1fb35e8b92de343f2a8--domain'])

        backlog_hosts = []
        for host in self.read_table('//home/antispam/export/skk/markup_queue/low_priority', yt_token):
            if host['owner'] not in checked_hosts:
                backlog_hosts.append(host['owner'])

        return backlog_hosts

    def backlog_distribution(self, remaining_assignees, yt_token=None, st_token=None):
        from yt.wrapper.errors import YtHttpResponseError

        try:
            backlog_hosts = self.backlog_unique(yt_token, st_token)
        except YtHttpResponseError:
            return {}

        selection = OrderedDict()
        for assignee in remaining_assignees:
            cutoff = LIMIT - len(remaining_assignees[assignee])
            cutoff = cutoff if cutoff <= len(backlog_hosts) else len(backlog_hosts)
            shuffle(backlog_hosts)
            for i in reversed(range(cutoff)):
                selection.setdefault(assignee, []).append(backlog_hosts.pop(i))

        return selection

    def hosts_distribution(self, yt_token=None, st_token=None):
        users_workload = self.get_users_workload(token=st_token)
        hosts = [host for host in self.read_table(YT_READ_PATH + str(datetime.now().date()), yt_token)]
        assignees = self.get_available_assignees(token=st_token)
        full_check_needed = [host['owner'] for host in hosts if host['full_check']]
        base_check_needed = [host['owner'] for host in hosts if host['base_check'] is not None and host['base_check']]

        selection = OrderedDict(full_check_needed=full_check_needed, base_check_needed=base_check_needed)
        for assignee in assignees:
            cutoff = LIMIT
            if assignee in users_workload:
                cutoff = LIMIT - users_workload[assignee] if users_workload[assignee] < LIMIT else 0
            cutoff = cutoff if cutoff <= len(hosts) else len(hosts)
            shuffle(hosts)
            for i in reversed(range(cutoff)):
                selection.setdefault(assignee, []).append(hosts.pop(i)['owner'])

        if len(hosts) > 0:
            selection['other'] = [host['owner'] for host in hosts]

        remaining_assignees = {assignee: [] for assignee in assignees
                               if assignee not in selection and assignee != 'other'}
        for assignee, sites in selection.items():
            if len(sites) < LIMIT:
                remaining_assignees[assignee] = sites

        if len(remaining_assignees) > 0:
            selection['backlog'] = self.backlog_distribution(remaining_assignees, yt_token, st_token)

        return selection

    @staticmethod
    def create_comment(issue_key, token=None):
        url = 'https://st-api.yandex-team.ru/v2/issues/{}/comments'.format(issue_key)
        headers = {'Authorization': 'OAuth {}'.format(token)}
        comment = '**Информацию в ТАСС игнорируем. Размечаем так, буд-то её нет (если она есть). Т.е размечаем полноценно.**'

        post(url, data=dumps({'text': comment}, ensure_ascii=False).encode('utf-8'),
             headers=headers)

    def create_issue(self, assignee, site, full_check_list, base_check_list, tags=None, token=None):
        url = STARTREK_URL + '/issues'
        headers = {'Accept-Charset': 'UTF-8', 'content-type': 'application/json',
                   'Authorization': 'OAuth {}'.format(token)}

        base_check = ''
        if site in base_check_list:
            base_check = '!!Данный тикет допускается проверить визуально!!'
            if tags is None:
                tags = 'Визуально'
            else:
                tags = [tags, 'Визуально']

        summary = 'Разметка сайта - {}'.format(site)
        description = DESCRIPTION.format(site=site, base_check=base_check)

        issue_data = dict(queue='SITEMARKING',
                          summary=summary,
                          type={'name': 'Задача'},
                          description=description,
                          tags=tags,
                          assignee=assignee)

        issue_data['6151d1fb35e8b92de343f2a8--domain'] = site

        r = post(url, data=dumps(issue_data, ensure_ascii=False).encode('utf-8'),
                 headers=headers).json()

        if site in full_check_list:
            self.create_comment(r['key'], token=token)

    def amnesty_distribute(self, st_token=None, yt_token=None):
        assignees = self.get_available_assignees(token=st_token)
        selection = OrderedDict()
        unbound = []
        full_check_needed = []

        for record in self.read_table(YT_AMNESTY, token=yt_token, end_index=None,
                                      columns=['assessments', 'owner', 'full_check']):
            counter = 0
            if record['full_check']:
                full_check_needed.append(record['owner'])

            while counter < record['assessments']:
                assignee = choice(assignees)
                selection.setdefault(record['owner'], [])

                if assignee not in selection.get(record['owner']):
                    selection[record['owner']].append(assignee)
                    counter += 1

                if len(assignees) == len(selection.get(record['owner'])):
                    unbound.append(record['owner'])
                    counter += 1

        for site, asessors in selection.items():
            for asessor in asessors:
                self.create_issue(asessor, site, full_check_needed, [], tags='Амнистия', token=st_token)

        for site in unbound:
            self.create_issue(None, site, full_check_needed, [], tags='Амнистия', token=st_token)

        self.clear_table(YT_AMNESTY, token=yt_token)
        sleep(20)

    @staticmethod
    def collect_opened_issues(filter, token=None):
        from startrek_client import Startrek
        client = Startrek(useragent='market-quality-control',
                          base_url=STARTREK_URL, token=token)
        return client.issues.find(
            filter=filter, perScroll=100, scrollType='unsorted',
            scrollTTLMillis=20000
        )

    def retry_transaction(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            from startrek_client.exceptions import NotFound

            try:
                return func(*args, **kwargs)
            except NotFound:
                return func(transition='closed', *args, **kwargs)

        return wrapper

    @retry_transaction
    def cancel_issue(self, issue, transition='close', token=None):
        comment = 'Проверка автоматически отменена.'
        issue.transitions[transition].execute(comment=comment, resolution='fixed')

    def cancel_hosts_check(self, yt_token=None, st_token=None):
        table = '//home/antispam/export/skk/markup_queue/cancel'
        hosts = []
        for record in self.read_table(table, token=yt_token, end_index=None, columns=['owner']):
            hosts.append(record['owner'])

        filter_ = {'queue': 'SITEMARKING', 'resolution': 'empty()'}
        issues = [issue for issue in self.collect_opened_issues(filter=filter_, token=st_token)]
        for issue in issues:
            if '6151d1fb35e8b92de343f2a8--domain' in vars(issue)['_value'] and \
                    issue['6151d1fb35e8b92de343f2a8--domain'] in hosts:
                self.cancel_issue(issue, token=st_token)
        self.clear_table(table, token=yt_token)

    def create_modadvert_issue(self, assignee, offer, tags=None, token=None):
        url = STARTREK_URL + '/issues'
        headers = {'Accept-Charset': 'UTF-8', 'content-type': 'application/json',
                   'Authorization': 'OAuth {}'.format(token)}

        summary = 'Разметка модерации рекламы - {}'.format(offer['domain'])
        description = ADVERT_DESCRIPTION.format(site=offer['domain'],
                                                offer_url=offer['href'],
                                                title=offer['title'],
                                                body=offer['body'])

        issue_data = dict(queue='SITEMARKING',
                          summary=summary,
                          type={'name': 'Задача'},
                          description=description,
                          tags=tags,
                          assignee=assignee,
                          components=[105181])

        issue_data['6151d1fb35e8b92de343f2a8--domain'] = offer['domain']
        post(url, data=dumps(issue_data, ensure_ascii=False).encode('utf-8'),
             headers=headers).json()

    def modadvert_distribute(self, yt_token=None, st_token=None):
        table = '//home/antispam/export/skk/markup_queue/modadvert'
        assignees = self.get_available_assignees(token=st_token, grid=MODADVERT_GRID)
        users_workload = self.get_users_workload(token=st_token, components='105181')

        hosts = list(self.read_table(table, token=yt_token, end_index=None, columns=None))

        selection = OrderedDict()
        for assignee in assignees:
            cutoff = LIMIT
            if assignee in users_workload:
                cutoff = LIMIT - users_workload[assignee] if users_workload[assignee] < LIMIT else 0
            cutoff = cutoff if cutoff <= len(hosts) else len(hosts)
            shuffle(hosts)
            for i in reversed(range(cutoff)):
                selection.setdefault(assignee, []).append(hosts.pop(i))

        if len(hosts) > 0:
            selection['other'] = hosts

        remaining_assignees = {
            assignee: [] for assignee in assignees if assignee not in selection and assignee != 'other'
        }
        for assignee, sites in selection.items():
            if len(sites) < LIMIT:
                remaining_assignees[assignee] = sites

        self.clear_table(table, token=yt_token)
        return selection

    def on_execute(self):
        from yt.wrapper.errors import YtHttpResponseError

        token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]
        yql_token = self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key]

        try:
            self.cancel_hosts_check(yt_token=yql_token, st_token=token)
            self.amnesty_distribute(st_token=token, yt_token=yql_token)
        except YtHttpResponseError:
            pass

        modadvert = self.modadvert_distribute(yt_token=yql_token, st_token=token)
        for assignee, hosts in modadvert.items():
            if assignee != 'other' and assignee != 'backlog':
                for offer in hosts:
                    self.create_modadvert_issue(assignee, offer, token=token)
            elif assignee == 'other':
                for offer in hosts:
                    self.create_modadvert_issue(None, offer, token=token)

        distribution = self.hosts_distribution(yql_token, token)
        for assignee, hosts in distribution.items():
            if assignee != 'full_check_needed' and assignee != 'base_check_needed':
                if assignee != 'other' and assignee != 'backlog':
                    for site in hosts:
                        self.create_issue(assignee, site, distribution['full_check_needed'], distribution['base_check_needed'], token=token)
                elif assignee == 'other':
                    for site in hosts:
                        self.create_issue(None, site, distribution['full_check_needed'], distribution['base_check_needed'], token=token)
                elif assignee == 'backlog':
                    for user, sites in hosts.items():
                        for site in sites:
                            self.create_issue(user, site, distribution['full_check_needed'], distribution['base_check_needed'], tags='Бэклог', token=token)
