# -*- coding: utf-8 -*-
from collections import namedtuple
from datetime import datetime
import functools
import os

YT_DATA_TABLE = '//home/market-quality-control/robot-johnny/not_shop_failed_orders'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'


def execute_query(query, token=None):
    from yql.api.v1.client import YqlClient

    client = YqlClient(token=token)
    request = client.query(query)
    request.run()

    for result in request.get_results():
        template = namedtuple('Record', result.column_names)
        for record in result.get_iterator():
            yield template(*record)


def __handle_table_data(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return {record['supplier_id']: record['orders']
                for record in func(*args, **kwargs)}

    return wrapper


@__handle_table_data
def read_table(table_read_path, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)

    return client.read_table(yt.TablePath(table_read_path),
                             format='json', raw=False)


def write_table(table_write_path, table_data, token=None, proxy='hahn'):
    import yt.wrapper as yt

    client = yt.client.Yt(token=token, proxy=proxy)

    client.write_table(yt.TablePath(table_write_path, append=False),
                       table_data, raw=False)


def create_issue(supplier_id, orders, st_token=None):
    from startrek_client import Startrek

    client = Startrek(useragent='market-quality-control',
                      base_url=STARTREK_URL, token=st_token)

    supplier_data = orders[0]
    summary = 'Магазин {} (ID: {}): отмены в статусе не SHOP_FAILED, {}'.format(
        supplier_data.shop_name, supplier_id, str(datetime.now().date()))

    description = 'Отчет АБО: https://abo.market.yandex-team.ru/partner/{}\n'.format(supplier_id) +\
        'ЛК партнёра: https://partner.market.yandex.ru/supplier/{}/summary\n'.format(supplier_data.campaign_id) +\
        '\nСписок заказов:\n' +\
        '\n'.join(['{} - Подстатус отмены: {}'.format(order.order_id, order.order_substatus)
                   for order in orders])

    client.issues.create(
        queue='BQ',
        summary=summary,
        type={'name': 'Задача'},
        description=description,
        tags='не_shopfailed',
        components=83456,
        idMagazina=supplier_id,
        customerOrderNumber=', '.join([str(_.order_id) for _ in orders]),
        defectOrders=len(orders),
        customerEmail=supplier_data.contacts.email
    )


def run(st_token=None, yt_token=None):
    former_orders = read_table(YT_DATA_TABLE, token=yt_token)

    with open(os.path.join(os.path.dirname(__file__), 'query.yql'), 'r') as f:
        yql_query = f.read()

    groupby_supplier = {}
    for record in execute_query(yql_query, token=yt_token):
        groupby_supplier.setdefault(record.supplier_id, []).append(record)

    for supplier_id, orders in groupby_supplier.items():
        actual_orders = [order for order in orders if order.order_id not in
                         former_orders.get(supplier_id, [])]

        if len(actual_orders) > 0:
            actual_orders = [order.order_id for order in actual_orders]
            former_orders.setdefault(supplier_id, []).extend(actual_orders)

            create_issue(supplier_id, orders, st_token=st_token)

    former_orders = [{'supplier_id': supplier_id, 'orders': orders}
                     for supplier_id, orders in former_orders.items()]

    write_table(YT_DATA_TABLE, former_orders, token=yt_token)
