# -*- coding: utf-8 -*-
from __future__ import absolute_import, division, print_function, unicode_literals

from json import dumps
from re import split, sub
from requests import get, post
from datetime import datetime, timedelta
from collections import OrderedDict, namedtuple

from sandbox.sandboxsdk import environments
from sandbox import sdk2

from sandbox.projects.MarketQC.transfer_ocv_fraud_orders.query import (
    CHECKOUTER_ORDERS, SHOP_CONTACTS
)

YT_PROXY = 'hahn'
YT_PATH = '//home/clickhouse-kolkhoz/ch_hitinap/skk_processes/ocv_fraud_orders'

GRID = 'users/hitinap/statichnyjj-email-partnjorov-v-ocv'
STARTREK_URL = 'https://st-api.yandex-team.ru/v2'
LIMIT = 150


class Transmitter:

    @staticmethod
    def execute_query(query, config={}):
        from psycopg2.extras import NamedTupleCursor
        from psycopg2 import connect

        with connect(cursor_factory=NamedTupleCursor, **config).cursor() as cursor:
            cursor.execute(query)
            return cursor.fetchall()

    @staticmethod
    def execute_yql(query, token=None):
        from yql.api.v1.client import YqlClient

        client = YqlClient(token=token)
        request = client.query(query)
        request.run()

        for result in request.get_results():
            record = namedtuple('Record', result.column_names)
            return [record(*row) for row in result.get_iterator()]

    @staticmethod
    def collect_issues(filter_, token=None):
        from startrek_client import Startrek

        client = Startrek(useragent='python',
                          base_url=STARTREK_URL, token=token, timeout=1600)

        return client.issues.find(filter=filter_, perScroll=1000,
                                  scrollType='unsorted', scrollTTLMillis=20000)

    @staticmethod
    def _process_issues(issue_generator):
        order_pattern = namedtuple('Order', ['order_id', 'key', 'component_id'])

        for issue in issue_generator:
            if issue.customerOrderNumber is None:
                for fragment in split(r',|;|\s+|\\|/|-', issue.summary):
                    fragment = sub(r'[A-Za-z]|[А-Яа-я]', '', fragment)
                    if fragment.isdigit() and len(fragment) == 8:
                        yield order_pattern(order_id=int(fragment),
                                            key=issue.key,
                                            component_id=issue.components[0].id)
            else:
                for order_number in split(r',|;|\s+|\\|/|-', issue.customerOrderNumber):
                    order_number = sub(r'[^0-9]', '', order_number)
                    if order_number.isdigit():
                        yield order_pattern(order_id=int(order_number),
                                            key=issue.key,
                                            component_id=issue.components[0].id)

    def collect_OCV_complaints(self, token=None):
        time_range = (datetime.now() - timedelta(days=7)).date()
        filter_ = {'queue': 'OCV',
                   'components': ['95101', '91526', '90659', '91198'],
                   'created': {'from': str(time_range)}}

        return self._process_issues(self.collect_issues(filter_, token))

    def collect_BLUEMARKETORDER_complaints(self, token=None):
        time_range = (datetime.now() - timedelta(days=7)).date()
        filter_ = {'queue': 'BLUEMARKETORDER',
                   'components': '83045',
                   'summary': '[Виджет]',
                   'created': {'from': str(time_range)},
                   'resolution': ['fixed', 'problemSolved']}

        return self._process_issues(self.collect_issues(filter_, token))

    def data_introduce(self, orders_table, yql_token=None, st_token=None, config={}):
        shop_pattern = namedtuple('Shop', ['shop_id', 'shop_name', 'campaign_id', 'business_id', 'order_cnt',
                                           'contacts', 'manager', 'orders', 'summary_rate', 'ancestor'])
        shops_data = OrderedDict()
        orders_data = OrderedDict()

        for order in self.collect_BLUEMARKETORDER_complaints(st_token):
            orders_data[order.order_id] = order

        for order in self.collect_OCV_complaints(st_token):
            orders_data[order.order_id] = order

        for order in self.execute_query(CHECKOUTER_ORDERS.format(','.join([str(_) for _ in orders_data])),
                                        config=config):
            if order.order_id in orders_data:
                shops_data.setdefault(order.shop_id, {'orders': []})['orders'].append(
                    orders_data[order.order_id])

        matched_shops_data = []
        for row in self.execute_yql(SHOP_CONTACTS.format(','.join([str(_) for _ in shops_data])),
                                    yql_token):
            history_data = orders_table[row.shop_id] if row.shop_id in orders_table else {}
            orders = [order for order in shops_data[row.shop_id]['orders']
                      if order.order_id not in history_data.setdefault('orders', [])]

            if len(orders) > 0:
                summary_rate = len(orders)/row.order_cnt * 100
                ancestor = history_data.setdefault('ancestor', None)

                if datetime.now() - timedelta(days=3) > history_data.setdefault('created', datetime(2021, 1, 1)):
                    matched_shops_data.append(shop_pattern(shop_id=row.shop_id,
                                                           shop_name=row.shop_name,
                                                           campaign_id=row.campaign_id,
                                                           business_id=row.business_id,
                                                           order_cnt=row.order_cnt,
                                                           contacts=row.contacts,
                                                           manager=row.manager_fullname,
                                                           orders=orders,
                                                           summary_rate=summary_rate,
                                                           ancestor=ancestor))
        if len(matched_shops_data) > LIMIT:
            matched_shops_data.sort(key=lambda s: s.summary_rate, reverse=True)
            matched_shops_data = matched_shops_data[:LIMIT]

        return matched_shops_data


class TransferOcvFraudOrders(sdk2.Task):
    """Daily transfer fraud orders in OCV"""

    class Requirements(sdk2.Requirements):
        ram = 1024
        cores = 1
        disk_space = 128

        environments = [
            environments.PipEnvironment('yql'),
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('psycopg2-binary'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('startrek_client', version='2.5',
                                        custom_parameters=['--upgrade-strategy only-if-needed'])
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        description = 'Daily transfer fraud orders in OCV'
        max_restarts = 3
        dump_disk_usage = False
        fail_on_any_error = True

        oauth_token = sdk2.parameters.YavSecret('OAuth ST token',
                                                required=True)
        yql_token = sdk2.parameters.YavSecret('OAuth YQL token',
                                              required=True)

        with sdk2.parameters.Group('Database parameters') as db_settings:
            host = sdk2.parameters.String('Host', default='man-i9jmlo2jx3thalwg.db.yandex.net',
                                          required=True)
            port = sdk2.parameters.Integer('Port', default=6432, required=True)
            dbname = sdk2.parameters.String('Database', required=True)
            user = sdk2.parameters.String('User', required=True)
            password = sdk2.parameters.YavSecret('Password', required=True)
            sslmode = sdk2.parameters.String('sslmode', default='require',
                                             required=True)

    @staticmethod
    def _fetch_email_exceptions(token=None):
        url = 'https://wiki-api.yandex-team.ru/_api/frontend/%s/.grid' % GRID
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        response = get(url, headers=headers, timeout=1200).json()

        emails = OrderedDict()
        for row in response['data']['rows']:
            columns = (row[0]['raw'], row[1]['raw'])
            shops = [sub(r'[^0-9]', '', shop_id) for shop_id in columns[0].split(',')]
            emails.update({int(shop_id): sub(r'\s+', '', columns[1]) for shop_id in shops})
        return emails

    @staticmethod
    def _save_orders_table(table, token=None):
        import yt.wrapper.client as wrapper

        client = wrapper.Yt(token=token, proxy=YT_PROXY)
        client.write_table('<append=%true>' + YT_PATH, table, raw=False)

    @staticmethod
    def __handle_orders_data(data_iterator):
        orders_table = OrderedDict()
        for row in data_iterator:
            created = datetime.strptime(row['created'], '%Y-%m-%d')
            orders_table.setdefault(row['shop_id'], {'ancestor': row['key'],
                                                     'created': created,
                                                     'orders': row['orders']})

            if created > orders_table[row['shop_id']]['created']:
                orders_table[row['shop_id']].update({'created': created,
                                                     'ancestor': row['key']})

            orders_table[row['shop_id']]['orders'] += row['orders']

        return orders_table

    def _get_orders_table(self, token=None):
        import yt.wrapper.client as wrapper

        client = wrapper.Yt(token=token, proxy=YT_PROXY)
        return self.__handle_orders_data(client.read_table(YT_PATH))

    @staticmethod
    def _build_connections(orders, ancestor):
        order_keys = [order.key for order in orders]
        connections = [{'relationship': 'is parent task for', 'issue': key}
                       for key in order_keys]
        if ancestor is not None:
            connections.append({'relationship': 'relates', 'issue': ancestor})

        return connections

    @staticmethod
    def _order_distribution(orders):
        order_groups = {'Покупатель не получил заказ, а статус заказа Доставлен': [],
                        'Покупатель не согласен с причиной отмены заказа': [],
                        'Покупатель не согласен с переносом даты доставки': [],
                        'Покупатель не согласен с изменением состава заказа': []}

        for order in orders:
            if order.component_id == '95101' or order.component_id == '83045':
                order_groups['Покупатель не получил заказ, а статус заказа Доставлен'].append(order.order_id)
            elif order.component_id == '90659':
                order_groups['Покупатель не согласен с причиной отмены заказа'].append(order.order_id)
            elif order.component_id == '91198':
                order_groups['Покупатель не согласен с переносом даты доставки'].append(order.order_id)
            elif order.component_id == '91526':
                order_groups['Покупатель не согласен с изменением состава заказа'].append(order.order_id)

        return order_groups

    def create_issue(self, shop_data, token=None):
        headers = {'Authorization': 'OAuth %s' % token,
                   'content-type': 'application/json', 'Accept-Charset': 'UTF-8'}
        URL = STARTREK_URL + '/issues'

        summary = 'Магазин {shopname} (ID: {shop_id}): потенциальный фрод со статусом заказа, {date}'.format(
            shopname=shop_data.shop_name, shop_id=shop_data.shop_id, date=datetime.now().strftime('%Y-%m-%d'))

        connections = self._build_connections(shop_data.orders, shop_data.ancestor)
        order_groups = self._order_distribution(shop_data.orders)

        description = 'Список заказов:\n'
        if len(order_groups['Покупатель не получил заказ, а статус заказа Доставлен']) > 0:
            description += '**Покупатель не получил заказ, а статус заказа Доставлен:**\n' +\
                '\n'.join([str(_) for _ in order_groups['Покупатель не получил заказ, а статус заказа Доставлен']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с причиной отмены заказа']) > 0:
            description += '**Покупатель не согласен с причиной отмены заказа:**\n' +\
                '\n'.join([str(_) for _ in order_groups['Покупатель не согласен с причиной отмены заказа']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с переносом даты доставки']) > 0:
            description += '**Покупатель не согласен с переносом даты доставки:**\n' +\
                '\n'.join([str(_) for _ in order_groups['Покупатель не согласен с переносом даты доставки']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с изменением состава заказа']) > 0:
            description += '**Покупатель не согласен с изменением состава заказа:**\n' +\
                '\n'.join([str(_) for _ in order_groups['Покупатель не согласен с изменением состава заказа']])

        comment = 'Отчет АБО: https://abo.market.yandex-team.ru/shop/{0}/report\n'.format(shop_data.shop_id) +\
            'ЛК партнёра: https://partner.market.yandex.ru/shop/{0}/summary\n'.format(shop_data.campaign_id) +\
            'Менеджер магазина: {0}\n'.format(shop_data.manager)

        comment += '\nСписок заказов:\n'
        if len(order_groups['Покупатель не получил заказ, а статус заказа Доставлен']) > 0:
            comment += '**Покупатель не получил заказ, а статус заказа Доставлен:**\n' +\
                '\n'.join(['((https://ow.market.yandex-team.ru/order/{0} {0}))'.format(_)
                           for _ in order_groups['Покупатель не получил заказ, а статус заказа Доставлен']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с причиной отмены заказа']) > 0:
            comment += '**Покупатель не согласен с причиной отмены заказа:**\n' +\
                '\n'.join(['((https://ow.market.yandex-team.ru/order/{0} {0}))'.format(_)
                           for _ in order_groups['Покупатель не согласен с причиной отмены заказа']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с переносом даты доставки']) > 0:
            comment += '**Покупатель не согласен с переносом даты доставки:**\n' +\
                '\n'.join(['((https://ow.market.yandex-team.ru/order/{0} {0}))'.format(_)
                           for _ in order_groups['Покупатель не согласен с переносом даты доставки']]) +\
                '\n\n'
        if len(order_groups['Покупатель не согласен с изменением состава заказа']) > 0:
            comment += '**Покупатель не согласен с изменением состава заказа:**\n' +\
                '\n'.join(['((https://ow.market.yandex-team.ru/order/{0} {0}))'.format(_)
                           for _ in order_groups['Покупатель не согласен с изменением состава заказа']])

        issue_data = {'queue': 'OCV',
                      'summary': summary,
                      'type': {'name': 'Задача'},
                      'description': description,
                      'components': 'DBS: потенциальный фрод со статусом заказа ежедневный',
                      'tags': 'ежедневный_тикет',
                      'links': connections,
                      'idMagazina': shop_data.shop_id,
                      'customerEmail': self._email_exceptions[shop_data.shop_id]
                      if shop_data.shop_id in self._email_exceptions else shop_data.contacts.email,
                      'merchant': shop_data.shop_name,
                      'businessIds': shop_data.business_id,
                      'customerOrderNumber': ', '.join(
                          [str(order.order_id) for order in shop_data.orders]),
                      'comment': comment}

        response = post(URL, data=dumps(issue_data, ensure_ascii=False).encode('utf-8'),
                        headers=headers).json()

        return response['key']

    def on_execute(self):
        # from sandbox.projects.MarketQC.transfer_ocv_fraud_orders.OrderCancelNotShopFailedStatus import run
        # from sandbox.projects.MarketQC.transfer_ocv_fraud_orders.without_final_status import run_upload

        st_token = self.Parameters.oauth_token.data()[self.Parameters.oauth_token.default_key]
        yql_token = self.Parameters.yql_token.data()[self.Parameters.yql_token.default_key]

        config = dict(host=self.Parameters.host,
                      port=self.Parameters.port,
                      dbname=self.Parameters.dbname,
                      user=self.Parameters.user,
                      password=self.Parameters.password.data()[self.Parameters.password.default_key],
                      sslmode=self.Parameters.sslmode)

        self._email_exceptions = self._fetch_email_exceptions(st_token)
        orders_table = self._get_orders_table(yql_token)
        transmitter = Transmitter()

        unloaded_newly_orders = []
        for shop_data in transmitter.data_introduce(orders_table, yql_token,
                                                    st_token, config):
            issue_key = self.create_issue(shop_data, st_token)
            unloaded_newly_orders.append({'key': issue_key,
                                          'created': datetime.now().strftime('%Y-%m-%d'),
                                          'orders': [order.order_id for order in shop_data.orders],
                                          'shop_id': shop_data.shop_id})

        self._save_orders_table(unloaded_newly_orders, yql_token)
        # Задача-паразит для генерации тикета проверки на заказы в статусе отмены не SHOP_FAILED
        # run(st_token=st_token, yt_token=yql_token)
        # run_upload(yt_token=yql_token, st_token=st_token)
